
/**
 * Title:        STEP test with SBX Test Card<p>
 * Description:  Code for mfg test of the STEP family of boards,
 * to test the SBX connector using the SBX Test card.<p>
 * Copyright:    Copyright (c) B Boyes<p>
 * Company:      Systronix<p>
 * @author B Boyes
 * @version 1.0
 */

/** Test a STEP, STEP+ or STEP.IR board
 * The SBX Test card has a TPIC open drain register U1, written with WR and CS1,
 * which drives the following signals. The INTO and INT1 lines may be read with
 * with CS1 and RD through register U6. Here are the output register bits.
 * <UL>
 * <LI> buzzer bit 7
 * <LI> green LED bit 6
 * <LI> red LED bit 5
 * <LI> MINT1 bit 1
 * <LI> MINT0 bit 0
 * </UL>
 * The MINT1 and MINT0 bits are all that can be read with U6.
 *
 * The test card also has an SRAM which is written with CS0 and WR, and read
 * with CS0 and RD. SBX only has three address lines, so only the lower three
 * bits of the write and read address are significant.
 *
 * On STEP, the SBX CS0 and CS1 are decoded at addresses:
 * 0x3800[6or7]X = CS0 where X can be 0-F, i.e. 0x380060 - 0x38007F
 * 0x3800[8or9]X = CS1 where X can be 0-F, i.e. 0x380080 - 0x38009F
 */

import java.util.*;
import java.io.*;
import java.lang.Integer;
import com.dalsemi.system.*;
import com.dalsemi.system.BitPort;
import com.dalsemi.onewire.OneWireAccessProvider;
import com.dalsemi.onewire.*;
import com.dalsemi.onewire.adapter.*;
import com.dalsemi.onewire.container.*;
//import com.dalsemi.onewire.container.ScratchPad;
import com.dalsemi.onewire.container.OneWireContainer10;

public class Step {

  static DSPortAdapter pa;
  static OneWireContainer09 device9;  // reference to the container we'll create
  static byte [] state09;     // holds the state of the device accessed through container
  static final int sbxCs0 = 0x380060;   // the SRAM on sbx test
  static final int sbxCs1 = 0x380080;   // the output and input registers
  static DataPort sbx2 = new DataPort ();
  static byte outData, inpData;   // bytes of I/O data at sbxCs1
  // These values are the actual bits in the output register
  // MINT1 and MINT0 also appear in the input register
  static final byte GREEN = 0x40;
  static final byte RED = 0x20;
  static final byte BUZZ = (byte) 0x80;   // high bit set requires typecast... interesting
  static final byte MINT0 = 0x01;
  static final byte MIN1 = 0x02;
  static int errorCount = 0;      // how many tests returned an error
  // this array is used to display hex values
  static final char[] digit = {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f'};
  byte[] data = new byte[100];

  static boolean debug = false;

  /** default constructor */
  public void step() {
  }

  /** main() calls a series of tests on the SBX connector.
   * Each test is implemented as its own method.
   * Each test is self contained, emits a message, and returns a pass/fail value.
   */
  public static void run( byte stretch, int iterate) {
    // Step step1 = new Step();
    boolean progress = true;  // receives the go/no-go return from the tests
    int timesThrough = 0;

    byte data = 0;

    BitPort p5B0 = new BitPort (BitPort.Port5Bit0);
    BitPort p3B5 = new BitPort (BitPort.Port3Bit5);

    outData |= BUZZ;
    outData |= RED;
    outData |= GREEN;
    writeOut();
    waitWhile (500L);
    outData &= ~BUZZ;   // turn off the buzzer & LEDs
    outData &= ~RED;
    outData &= ~GREEN;
    writeOut();

    do {
      System.out.print("OneWire Test-");
      progress = owTest();
      if (progress) {
        System.out.println(" -- OK");
        led (GREEN);
      }
      else errorCount++;

      System.out.print("Address Test with stretch=" + stretch + " ");
      progress = adrTest(stretch);
      if (progress) {
        System.out.println(" -- OK");
        led (GREEN);
      }
      else {
        errorCount++;
        System.out.println(" -- FAILED!");
        led (RED);
      }

      System.out.print("Memory Test with stretch=" + stretch + " ");
      progress = memTest( stretch );
      if (progress) {
        System.out.println(" -- OK");
        led (GREEN);
      }
      else {
        errorCount++;
        System.out.println(" -- FAILED!");
        led (RED);
      }


      System.out.print("Interrupt line test ");
      progress = intTest( );
      if (progress) {
        System.out.println(" -- OK");
        led (GREEN);
      }
      else {
        errorCount++;
        System.out.println(" -- FAILED!");
        led (RED);
      }

      if (errorCount > 0) {
        // turn buzzer and red LED on
        led (RED);
        outData |= BUZZ;
        writeOut();
        waitWhile (500L);
        outData &= ~BUZZ;   // buzzer off
        writeOut();
      }
      else {

      }
      System.out.print(++timesThrough + " iterations, ");
      System.out.print(errorCount + " failures");

    // do it forever or the number of times we were told
    } while ( (timesThrough < iterate) || (0 == iterate) );
    // end while

    if (0==errorCount) System.out.println(" -- ALL OK");
    else System.out.println(" -- TEST FAILURE");
  } // end run

  //-------------------------------------------------------
  /** Test the one-wire net on SBX OPT0, pin 30.
   * On STEP boards TINI External 1-Wire is tied to OPT0.
   * On SBX Test we have a DS2502, connected through SBX Test jumper JP1.
   * DS2502 is One Wire Family Code 09
   */
  private static boolean owTest () {
  boolean flag = true;

  try {
    pa = OneWireAccessProvider.getDefaultAdapter();
  }
  catch (OneWireException e) {
    System.out.print("Can't find 1Wire adapter " + e);
    // need to bail out if no adapter -- how?
    return false;
  }

  pa.targetFamily(0x09);     // Search only for the DS2502 family code.
  try {
    pa.beginExclusive(true);
    device9 = (OneWireContainer09) pa.getFirstDeviceContainer();
    if (null==device9 ) {
      System.out.println("Could not find DS2502 - is Jumper JP1 installed?");
      flag = false;
    }
    else {
      System.out.print("Found " + device9.getName()
                        + "/" + device9.getAlternateNames());
      //System.out.println (device9.getDescription());
      }

    pa.endExclusive();
  }
  catch (OneWireException e) {
    System.out.println("OneWire error " + e);
    flag = false;
  }
  return flag;
  }


  //-------------------------------------------------------
  /** Test the SBX INT0 and INT1 lines
   * MINT0 is pin 14, MINT1 is pin 12
   *
   */
  private static boolean intTest () {
  boolean flag = true;
  byte wdata, rdata = 0;


  // use at least one stretch to be sure this write works
  sbx2.setStretchCycles((byte)2);
  sbx2.setAddress (sbxCs1);    // input and output registers

  for (wdata=0; wdata < 0x04; wdata++) {
    if (flag) System.out.print(wdata);   // only print this if no errors to print

    outData = (byte) ((outData & 0xfc) | wdata );   // or in lower 2 bits of int data
    // could use writeOut and then just compare inpData and outData
    // no need for the try-catch here
    try {
      sbx2.write (outData);
      rdata = (byte) (sbx2.read() & 0x03) ;         // just compare intx bits
      }
    catch (Exception e) {
      System.out.println("Internal sbx address error " + e);
      flag = false;
      }
    if (wdata != rdata) {
      System.out.println("Interrupt error " + " wrote:" + hex(wdata) + " read:" + hex(rdata) );
      flag = false;
    }
  } // end for

  outData &= 0xfc;      // clear interrupt bits
  writeOut();

  return flag;
  }

  /** Test the SBX reset line pin 5
   * Turn the buzzer on then turn it off by resetting its register
   * We can also try to write to the INT lines and read them, then clear them
   * with by resetting register U1.
   */
  public boolean rstTest () {
  boolean flag = true;

  return flag;
  }

  //-------------------------------------------------------
  /** Test the SBX address and data lines
   * <BR>
   * MA0 is pin 11, MA1 is pin 9, MA2 is pin 7
   * MD7 = pin 19, MD6 = pin 21 ... MD0 = pin 33
   * MSC0 = pin 22
   * Contents of the data RAM is random on powerup.
   * Write to 0xF addresses, 0-7 and 8-F are mirrors of each other since...
   * ... we only have 8 addresses on an SBX board.
   */
  private static boolean adrTest (byte stretch) {
  boolean flag = true;
  int addr;
  byte wdata;   // data we'll write
  byte rdata=0;   // data we read back

  sbx2.setStretchCycles(stretch);

  for (addr=0; addr < 0x07; addr++) {
    sbx2.setAddress (sbxCs0 + addr);    // access all 8 addresses twice
    wdata = (byte) (0xa0 + addr);       // lsb of data is the address 0-F
    try {
      sbx2.write (wdata);
      rdata = (byte) sbx2.read();
      }
    catch (Exception e) {
      System.out.println("Internal sbx address error " + e);
      flag = false;
      }
    if (wdata != rdata) {
      System.out.println("Test1 error at " + hex( (byte)addr) + " wrote:" + hex(wdata) + " read:" + hex(rdata) );
      flag = false;
    }
  }

  // now just read the data we wrote previously
  for (addr=0; addr < 0x07; addr++) {
    sbx2.setAddress (sbxCs0 + addr);    // access all 8 addresses twice
    wdata = (byte) (0xa0 + addr);       // lsb of data is the address 0-F
    try {
      rdata = (byte) sbx2.read();
      }
    catch (Exception e) {
      System.out.println("Internal sbx address error " + e);
      flag = false;
      }
    if (wdata != rdata) {
      System.out.println("Test2 error at " + hex( (byte)addr) + " wrote:" + hex(wdata) + " read:" + hex(rdata) );
      flag = false;
    }
  }


  return flag;
  } // end adrTest


  //-------------------------------------------------------
  /** Test the SBXtest card memory
   * <BR>
   * MA0 is pin 11, MA1 is pin 9, MA2 is pin 7
   * MD7 = pin 19, MD6 = pin 21 ... MD0 = pin 33
   * MSC0 = pin 22
   * Contents of the data RAM is random on powerup.
   * Write to 0xF addresses, 0-7 and 8-F are mirrors of each other since...
   * ... we only have 8 addresses on an SBX board.
   */
  private static boolean memTest ( byte stretch ) {
  boolean flag = true;
  int addr;
  byte wdata;   // data we'll write
  byte rdata=0;   // data we read back

  sbx2.setStretchCycles(stretch);

  // ----- Test1 -----
  for (addr=0; addr < 0x08; addr++) {
    sbx2.setAddress (sbxCs0 + addr);    // access all 8 addresses
    if (flag) System.out.print(addr);   // only print this if no errors to print
    try {
      // write all possible data values to each address
      wdata = 0;
      do {
        sbx2.write (wdata);
        rdata = (byte) sbx2.read();
        if (debug) System.out.println("Mem1 @" + hex( (byte)addr) + " " + hex(wdata) + "/" + hex(rdata) );
        if (wdata != rdata) {
          System.out.println("Mem1 error @" + hex( (byte)addr) + " " + hex(wdata) + "/" + hex(rdata) );
          flag = false;
        }
        wdata ++;
      } while (wdata != 0);  // end while
    }
    catch (Exception e) {
      System.out.println("Internal sbx address error " + e);
      flag = false;
    }
  } // end of for

  // ----- Test2 -----
    // now write sequential data to sequential addresses
    // then go back and see if right data is in the correct place
    try {
      for (addr=0; addr < 0x08; addr++) {
        sbx2.setAddress (sbxCs0 + addr);    // access all 8 addresses twice
        wdata = (byte) (0xf8 + addr);       // ls 3 bits of data is the address 0-7
        sbx2.write (wdata);
      }
      // now just read the data we wrote previously
      for (addr=0; addr < 0x08; addr++) {
        sbx2.setAddress (sbxCs0 + addr);    // access all 8 addresses twice
        wdata = (byte) (0xf8 + addr);       // ls 3 bits of data is the address 0-7
        rdata = (byte) sbx2.read();
        if (debug) System.out.println("Mem2 @" + hex( (byte)addr) + " " + hex(wdata) + "/" + hex(rdata) );
        if (wdata != rdata) {
          System.out.println("Mem2 error @" + hex( (byte)addr) + " " + hex(wdata) + "/" + hex(rdata) );
          flag = false;
        }
      }
    }
    catch (Exception e) {
      System.out.println("Internal sbx address error " + e);
      flag = false;
    }


  return flag;
  } // end adrTest




  //-------------------------------------------------------
  /** wait howLong number of milliseconds
   */
  private static void waitWhile (long howLong) {
    try { Thread.sleep(howLong); }
    catch (InterruptedException e) {
    System.out.println("waitWhile error " + e);
    }
  } // end waitWhile

  //-------------------------------------------------------
  /** turn on red or green LED
   */
  private static void led (byte color) {

    if (GREEN==color) {
      // green
      outData &= ~RED;    // 1101 1111 mask off bit 5
      outData |= GREEN;    // 0100 0000 turn on bit 6 (green)
    }
    else {
      // red
      outData &= ~GREEN;    // 1011 1111 mask off bit 6
      outData |= RED;    // 0100 0000 turn on bit 5 (red)
    }
    writeOut();           // update the outputs
  } // end led

  //-------------------------------------------------------
  /** write outData to sbx CS1, the output register address
   * also read the input register and store in inpData
   */
  private static boolean writeOut () {
  boolean flag=true;
    try {
      sbx2.setAddress (sbxCs1);
      sbx2.write (outData);
      inpData = (byte) sbx2.read ();
    }
    catch ( IllegalAddressException e ){
      System.out.println ("dataPort write error " + e);
      errorCount++;
      flag = false;
    }

    return flag;
  } // end writeOut

  //-------------------------------------------------------
  /** from Jac Kersing
   * converts a byte array len elements long to hex representation in a string
   */
  private void display(byte[] data, int len) {
    for (int i = 0; i < len; i++) {
      System.out.print(hex(data[i])+" ");
    }
    System.out.println("");
  }

  // returns a string of hex representation when given a single byte value
  private static String hex(byte b) {
    return "" + digit[(b >> 4)&0x0f] + digit[b&0x0f];
  }

  //-------------------------------------------------------
  /** Main
   * pass # stretch cycles to use, # of times to iterate
   */
  public static void main (String[] args) {

    int stretch = 1;    // default stretch cycles
    int iterate = 0;    // times to loop through tests, 0 means forever

    Step step1 = new Step();

    try {

      System.out.println("STEP SBX Test program rev 2000 Dec 20 bab");
      System.out.println("usage: java step.tini [stretch_cycles] [iterations]");
      System.out.println("  stretch_cycles are used in the sbx access tests (0-7, default is 1)");
      System.out.println("    stretch can be 0-7, default is 1");
      System.out.println("  iterations are the number of times to repeat the tests");
      System.out.println("    iterations can be 0-2,000,000,000 -- default (0) is forever");
      System.out.println("Be sure SBX Test Fixture is plugged into the STEP Board");
      // we want 2 args which would make args.length == 2
      if (args.length < 2) {
        // no iteration entered so use default
        iterate = 0;
      }
      else {
        // try to convert args[1]
        iterate = step1.convertArg (args[1]);
        if ((Integer.MAX_VALUE == iterate) | (iterate < 0) ) {iterate = 0;}   // default to forever
      }
      if (args.length < 1) {
        // no stretch entered so use default
        stretch = 1;
      }
      else {
        // try to convert args[0]
        stretch = step1.convertArg (args[0]);
        if ((stretch > 7) | (stretch < 0) ) stretch = 1;   // default to 1
      }
      System.out.println("Preparing to use stretch of " + stretch + " and iterate " + iterate + " times");
    }
    catch (Exception e) {
	System.out.println("Exception: " + e.getMessage());
    }

    try {
      step1.run( (byte) stretch, iterate);
    }
    catch (Exception e) {
      System.out.println("Exception: " + e.getMessage());
    }

  } // end main

  /** try to convert string to signed integer
   * returns the int value or MAX_VALUE if unable
   */
  private int convertArg(String arg) {
    int value = Integer.MAX_VALUE;
    try {
      value = Integer.parseInt (arg);
    }
    catch (NumberFormatException nfe) {
      System.out.println(nfe.getMessage() + " : " + nfe);
      nfe.printStackTrace();
    }
    catch (Exception e) {
      System.out.println("Exception: " + e.getMessage());
    }

    return value;
  } // end convertArg

}