//================================================================
//                Copyright (c) 2001 Systronix
//                     All Rights Reserved
//================================================================

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import java.lang.System;

import javax.microedition.io.Connection;
import javax.microedition.io.Connector;
import javax.microedition.io.StreamConnection;

public class EthernetWraparoundTest {

    private static final String dataPrefix = "Ethernet test message number ";
    private static final String testName   = "Ethernet wraparound test";
    private static final byte SENTINEL_VALUE = -1;

    private boolean verbose;

    public EthernetWraparoundTest(boolean verbose){
        this.verbose = verbose;
    }  // constructor EthernetWraparoundTest

    // This is intended for host-based testing with CLDC reference
    // implementations or for standalone testing.
    //
    public static void main(String [] args) {
        EthernetWraparoundTest t = new EthernetWraparoundTest(true);
        try {
            t.run("192.168.10.13", "8188", 20);
        } catch (IOException ioe) {
                System.out.print(testName);
                System.out.print(" failed due to IOException ");
                System.out.println(ioe);
        }
    }

    public boolean run(String hostIP, String hostPort, int iterations) throws IOException {

        boolean passed = true;

        // Bail out if something went wrong setting up the socket or streams

        if (verbose) {
            System.out.print("Connecting to host ");
            System.out.print(hostIP);
            System.out.print(" port number ");
            System.out.println(hostPort);
        }

        String connURL = "socket://" + hostIP + ":" + hostPort;

        StreamConnection socket  = (StreamConnection) Connector.open("socket://" + hostIP + ":" + hostPort);

        if (verbose) {
            System.out.println("Opened socket");
        }

        InputStream  inStream  = socket.openInputStream();

        if (verbose) {
            System.out.println("Opened input stream");
        }

        OutputStream outStream = socket.openOutputStream();

        if (verbose) {
            System.out.println("Opened output stream");
        }

        socket.close();

        // Run the tests

        passed = echoTest(inStream, outStream, iterations);

        return passed;

    } // method run

    /**
     * Stream echo test; write enough bytes to fill buffers, try it a few times
     * return pass/fail.
     */
    private boolean echoTest(InputStream inStream, OutputStream outStream, int iterations) {

        byte [] outputBuf;

        boolean passed = true;

        try {

            // Write a few buffers to output and compute a checksum

            int outputSum = 0;
            int outputBytes = 0;

            for (int i=0; i < iterations; i++) {
                outputBuf = (dataPrefix + Integer.toString(i)).getBytes();
                outStream.write(outputBuf);

                for (int j=0; j < outputBuf.length; j++) {
                    outputSum += outputBuf[j];
                    outputBytes++;
                }
                outputBuf = null;
            }

            if (verbose) {
              System.out.println("Wrote " + outputBytes + " bytes to socket");
              System.out.println("Checksum is " + outputSum);
            }

            // Sentinel value to instruct server that we're done

            outStream.write(0xFF);

            outStream.flush();
            outStream.close();

            if (verbose) {
              System.out.println("Flushed and closed output stream");
            }

            // Read the input until it is exhausted and compute its
            // checksum.

            int inputSum = 0;
            int inputBytes = 0;

            byte [] inBuf = new byte [256];
            int numBytes;

            boolean sentinelFound = false;

            while((numBytes = inStream.read(inBuf)) != -1) {
                for (int i=0; i < numBytes; i++) {

                    //if (verbose) {
                    //    System.out.println("Read byte number " + inputBytes + " with value " + inBuf[i]);
                    //}

                    if (inBuf[i] == SENTINEL_VALUE) {
                        sentinelFound = true;
                    } else {
                        inputSum += inBuf[i];
                        inputBytes++;    // Don't count sentinel
                    }
                }

                if (verbose) {
                    System.out.println("Read chunk with " + numBytes + " bytes");
                    System.out.println("Current checksum is " + inputSum);

                    if (sentinelFound) {
                        System.out.println("Detected sentinel byte");
                        break;
                    }
                }
            }

            if (verbose) {
              System.out.println("Read " + inputBytes + " bytes total from socket");
              System.out.println("Checksum is " + inputSum);
            }

            inStream.close();

            // Lengths must compare and bytes must compare

            if (inputBytes != outputBytes) {
                System.out.print(testName);
                System.out.println(" length miscompare; sent " + outputBytes + " received " + inputBytes);
                passed = false;

            } else if (inputSum != outputSum) {
                System.out.print(testName);
                System.out.println(" checksum miscompare; output " + outputSum + " input " + inputSum);
                passed = false;
            }

        } catch (IOException ioe) {
                System.out.print(testName);
                System.out.print(" failed due to IOException ");
                System.out.println(ioe);
                passed = false;
        }

        return passed;

    }

} // end ether test


