/******************************************************************************
 *
 * Copyright (c) 1999 Palm Computing, Inc. or its subsidiaries.
 * All rights reserved.
 *
 * File: Starter.c
 *
 *****************************************************************************/

#include <PalmOS.h>
#include "StarterRsc.h"


/***********************************************************************
 *
 *   Entry Points
 *
 ***********************************************************************/


/***********************************************************************
 *
 *   Internal Constants
 *
 ***********************************************************************/
#define appFileCreator				 'TB-2'
#define appVersionNum              0x00
#define appPrefID                  0x01
#define appPrefVersionNum          0x01

// Define the minimum OS version we support (3.0 for now).
#define ourMinVersion	sysMakeROMVersion(3,0,0,sysROMStageRelease,0)

// Define states:
#define	IDLE	0
#define	DISC	1
#define	CON_LAP	2
#define	IAS_Q	3
#define	CON_LMP	4
#define	DATA	5
#define	DISCON	6

// Misc constants
#define	RESULT_SIZE	128

unsigned char rlsap_temp_query[] = "\013Temperature\004lsap";

/***********************************************************************
 *
 *   Internal Structures
 *
 ***********************************************************************/
typedef struct 
	{
	UInt8 replaceme;
	} StarterPreferenceType;

typedef struct 
	{
	UInt8 replaceme;
	} StarterAppInfoType;

typedef StarterAppInfoType* StarterAppInfoPtr;


/***********************************************************************
 *
 *   Global variables
 *
 ***********************************************************************/
UInt16 irRef;
int rlsap = 0;
IrDeviceAddr devAddr;
IrConnect connection;
IrPacket packet;
unsigned char data[20];
IrIasQuery query;
int state = 0;
unsigned char result[RESULT_SIZE];
int count;

/***********************************************************************
 *
 *   Internal Functions
 *
 ***********************************************************************/
FieldPtr SetFieldTextStr( Boolean, UInt16, MemPtr);
void GetTemp(void);
static void IrCallback(IrConnect *, IrCallBackParms *);
static void IrIasCallBack(IrStatus);

/***********************************************************************
 *
 * FUNCTION:    RomVersionCompatible
 *
 * DESCRIPTION: This routine checks that a ROM version is meet your
 *              minimum requirement.
 *
 * PARAMETERS:  requiredVersion - minimum rom version required
 *                                (see sysFtrNumROMVersion in SystemMgr.h 
 *                                for format)
 *              launchFlags     - flags that indicate if the application 
 *                                UI is initialized.
 *
 * RETURNED:    error code or zero if rom is compatible
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static Err RomVersionCompatible(UInt32 requiredVersion, UInt16 launchFlags)
{
	UInt32 romVersion;

	// See if we're on in minimum required version of the ROM or later.
	FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
	if (romVersion < requiredVersion)
		{
		if ((launchFlags & (sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp)) ==
			(sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp))
			{
			FrmAlert (RomIncompatibleAlert);
		
			// Palm OS 1.0 will continuously relaunch this app unless we switch to 
			// another safe one.
			if (romVersion < ourMinVersion)
				{
				AppLaunchWithCommand(sysFileCDefaultApp, sysAppLaunchCmdNormalLaunch, NULL);
				}
			}
		
		return sysErrRomIncompatible;
		}

	return errNone;
}

/***********************************************************************
 *
 * FUNCTION:    GetObjectPtr
 *
 * DESCRIPTION: This routine returns a pointer to an object in the current
 *              form.
 *
 * PARAMETERS:  formId - id of the form to display
 *
 * RETURNED:    void *
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static void * GetObjectPtr(UInt16 objectID)
{
	FormPtr frmP;

	frmP = FrmGetActiveForm();
	return FrmGetObjectPtr(frmP, FrmGetObjectIndex(frmP, objectID));
}

/* SetFieldTextStr: set text from static string
 */
FieldPtr SetFieldTextStr( Boolean update, UInt16 field, MemPtr str) {
	MemHandle oldTxtH;
	MemHandle txtH;
	MemPtr txtP;
	FieldPtr fld;
	Boolean editable;
	
	txtH = MemHandleNew( StrLen(str) + 1);
	txtP = MemHandleLock( txtH );
	StrCopy( txtP, str );
	MemHandleUnlock( txtH );
	txtP = 0;
	
	fld = GetObjectPtr( field );
	oldTxtH = FldGetTextHandle( fld );
	
	editable = fld->attr.editable;
	fld->attr.editable = true;
	
	FldSetTextHandle( fld, txtH );
	if (update) {
		FldDrawField( fld );
	}
	fld->attr.editable = editable;
	
	if (oldTxtH != NULL) {
		MemHandleFree( oldTxtH );
	}
	
	return fld;
}

/* IrIasCallBack: callback for IAS functions
 */
static void IrIasCallBack(IrStatus status) {
	int type;
	
	if (query.retCode == IAS_RET_SUCCESS) {
		type = IrIAS_GetType(&query);
		
		if (type == IAS_ATTRIB_INTEGER && state == IAS_Q) {
			rlsap = IrIAS_GetIntLsap(&query);
			connection.rLsap = rlsap;
			packet.buff = data;
			packet.len = 0;
			SetFieldTextStr(true, MainStatusField, "Connecting...");
			if (IrConnectReq(irRef, &connection, &packet, 1) != IR_STATUS_PENDING) {
				SetFieldTextStr(true, MainStatusField, "Connect request failed");
				state = IDLE;
			} else {
				state = CON_LMP;
			}
		}
	} else {
		SetFieldTextStr(true, MainStatusField, "Device does not do TEMP");
		IrDisconnectIrLap(irRef);
		state = IDLE;
	}
}
	
	
/* IrCallback: all IR functions call this upon completion
 */
static void IrCallback(IrConnect *connect, IrCallBackParms *parms) {
	int i;
	
	switch(parms->event) {
		case LEVENT_STATUS_IND:
			break;

		case LEVENT_DISCOVERY_CNF:
			if (parms->deviceList->nItems > 0) {
				devAddr.u32 = parms->deviceList->dev[0].hDevice.u32;
			
				SetFieldTextStr(true, MainStatusField, "Connecting");
				
				if (IrConnectIrLap(irRef, devAddr) != IR_STATUS_PENDING) {
					SetFieldTextStr(true, MainStatusField, "Connection request failed");
					state = IDLE;
				} else {
					state = CON_LAP;
				}
			} else {
				// should retry here...
			}
			break;

		case LEVENT_LM_DISCON_IND:
			if (state == CON_LMP) {
				SetFieldTextStr(true, MainStatusField, "Connection refused");
			}
			break;

		case LEVENT_LAP_CON_CNF:
			// got LAP connection, now query for IAS Parameter
			query.queryBuf = rlsap_temp_query;
			query.queryLen = 17;
			query.resultBufSize = RESULT_SIZE;
			query.result = result;
			query.callBack = IrIasCallBack;
			if (IrIAS_Query(irRef, &query) != IR_STATUS_PENDING) {
				SetFieldTextStr(true, MainStatusField, "Lookup failed");
				IrDisconnectIrLap(irRef);
				state = IDLE;
			} else {
				state = IAS_Q;
			}
			break;

		case LEVENT_DATA_IND:
			SetFieldTextStr(true, MainStatusField, "Got data");
			SetFieldTextStr(true, MainTempField, parms->rxBuff);
			// disconnect
			state = DISC;
			if(IrDisconnectIrLap(irRef)!=IR_STATUS_PENDING) {
				SetFieldTextStr(true, MainStatusField, "Disconnect request failed");
			}
			break;

		case LEVENT_LM_CON_CNF:
			// connection up, get temperature
			SetFieldTextStr(true, MainStatusField, "Get data");
			data[0] = 0x01;
			packet.buff = data;
			packet.len = 1;
			if (IrDataReq(irRef, &connection, &packet) != IR_STATUS_PENDING) {
				SetFieldTextStr(true, MainStatusField, "Data request failed");
				state = IDLE;
			} else {
				state = DATA;
			}
			break;

		case LEVENT_LAP_DISCON_IND:
			if (state == CON_LAP) {
				SetFieldTextStr(true, MainStatusField, "Connect failed");
			} else {
				SetFieldTextStr(true, MainStatusField, "Disconnected");
			}
			break;
			
		case LEVENT_PACKET_HANDLED:
			break;
	}
}


/* GetTemp(): get the current temperature from TINI by IrDA
 */
void GetTemp() {
	SetFieldTextStr(true, MainStatusField, "Discover TINI");

	count = 0;	
	if (IrIsMediaBusy(irRef)) {
		SetFieldTextStr(true, MainStatusField, "IR media busy");
	} else {
		if (IrDiscoverReq(irRef, &connection) != IR_STATUS_PENDING) {
			SetFieldTextStr(true, MainStatusField, "Discovery request failed");
		}
	}
}	


/***********************************************************************
 *
 * FUNCTION:    MainFormInit
 *
 * DESCRIPTION: This routine initializes the MainForm form.
 *
 * PARAMETERS:  frm - pointer to the MainForm form.
 *
 * RETURNED:    nothing
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static void MainFormInit(FormPtr /*frmP*/)
{
	SetFieldTextStr(false, MainStatusField, "Initialized. Ready for update");
	SetFieldTextStr(false, MainTempField, "----");
}


/***********************************************************************
 *
 * FUNCTION:    MainFormDoCommand
 *
 * DESCRIPTION: This routine performs the menu command specified.
 *
 * PARAMETERS:  command  - menu item id
 *
 * RETURNED:    nothing
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static Boolean MainFormDoCommand(UInt16 command)
{
	Boolean handled = false;
   FormPtr frmP;

	switch (command)
		{
		case MainOptionsAboutTINITemp:
			MenuEraseStatus(0);					// Clear the menu status from the display.
			frmP = FrmInitForm (AboutForm);
			FrmDoDialog (frmP);					// Display the About Box.
			FrmDeleteForm (frmP);
			handled = true;
			break;

		}
	
	return handled;
}

/***********************************************************************
 *
 * FUNCTION:    MainFormHandleEvent
 *
 * DESCRIPTION: This routine is the event handler for the 
 *              "MainForm" of this application.
 *
 * PARAMETERS:  eventP  - a pointer to an EventType structure
 *
 * RETURNED:    true if the event has handle and should not be passed
 *              to a higher level handler.
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static Boolean MainFormHandleEvent(EventPtr eventP)
{
   Boolean handled = false;
   FormPtr frmP;

	switch (eventP->eType) 
		{
		case menuEvent:
			return MainFormDoCommand(eventP->data.menu.itemID);

		case frmOpenEvent:
			frmP = FrmGetActiveForm();
			MainFormInit( frmP);
			FrmDrawForm ( frmP);
			handled = true;
			break;

		case ctlSelectEvent:
			if (eventP->data.ctlEnter.controlID == MainGetTempButton) {
				GetTemp();
			}
			handled = true;
			break;

		default:
			break;
		
		}
	
	return handled;
}


/***********************************************************************
 *
 * FUNCTION:    AppHandleEvent
 *
 * DESCRIPTION: This routine loads form resources and set the event
 *              handler for the form loaded.
 *
 * PARAMETERS:  event  - a pointer to an EventType structure
 *
 * RETURNED:    true if the event has handle and should not be passed
 *              to a higher level handler.
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static Boolean AppHandleEvent(EventPtr eventP)
{
	UInt16 formId;
	FormPtr frmP;

	if (eventP->eType == frmLoadEvent)
		{
		// Load the form resource.
		formId = eventP->data.frmLoad.formID;
		frmP = FrmInitForm(formId);
		FrmSetActiveForm(frmP);

		// Set the event handler for the form.  The handler of the currently
		// active form is called by FrmHandleEvent each time is receives an
		// event.
		switch (formId)
			{
			case MainForm:
				FrmSetEventHandler(frmP, MainFormHandleEvent);
				break;

			default:
//				ErrFatalDisplay("Invalid Form Load Event");
				break;

			}
		return true;
		}
	
	return false;
}


/***********************************************************************
 *
 * FUNCTION:    AppEventLoop
 *
 * DESCRIPTION: This routine is the event loop for the application.  
 *
 * PARAMETERS:  nothing
 *
 * RETURNED:    nothing
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static void AppEventLoop(void)
{
	UInt16 error;
	EventType event;

	do {
		EvtGetEvent(&event, evtWaitForever);

		if (! SysHandleEvent(&event))
			if (! MenuHandleEvent(0, &event, &error))
				if (! AppHandleEvent(&event))
					FrmDispatchEvent(&event);

	} while (event.eType != appStopEvent);
}


/***********************************************************************
 *
 * FUNCTION:     AppStart
 *
 * DESCRIPTION:  Get the current application's preferences.
 *
 * PARAMETERS:   nothing
 *
 * RETURNED:     Err value 0 if nothing went wrong
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static Err AppStart(void)
{
    StarterPreferenceType prefs;
    UInt16 prefsSize;

	// Read the saved preferences / saved-state information.
	prefsSize = sizeof(StarterPreferenceType);
	if (PrefGetAppPreferences(appFileCreator, appPrefID, &prefs, &prefsSize, true) != 
		noPreferenceFound)
		{
		}

	// open Ir channel
	if (SysLibFind(irLibName, &irRef) != 0) {
		return 1;
	}
	if (IrOpen(irRef, 0) != 0) {
		return 1;
	}
	if (IrBind(irRef, &connection, IrCallback) != 0) {
		return 1;
	}
	IrSetConTypeLMP(&connection);
	
   	return errNone;
}


/***********************************************************************
 *
 * FUNCTION:    AppStop
 *
 * DESCRIPTION: Save the current state of the application.
 *
 * PARAMETERS:  nothing
 *
 * RETURNED:    nothing
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static void AppStop(void)
{
   StarterPreferenceType prefs;

	// Write the saved preferences / saved-state information.  This data 
	// will be backed up during a HotSync.
	PrefSetAppPreferences (appFileCreator, appPrefID, appPrefVersionNum, 
		&prefs, sizeof (prefs), true);

	// close IR
	if (irRef != 0) {
		IrClose(irRef);
	}
			
	// Close all the open forms.
	FrmCloseAllForms ();
}


/***********************************************************************
 *
 * FUNCTION:    StarterPalmMain
 *
 * DESCRIPTION: This is the main entry point for the application.
 *
 * PARAMETERS:  cmd - word value specifying the launch code. 
 *              cmdPB - pointer to a structure that is associated with the launch code. 
 *              launchFlags -  word value providing extra information about the launch.
 *
 * RETURNED:    Result of launch
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static UInt32 StarterPalmMain(UInt16 cmd, MemPtr /*cmdPBP*/, UInt16 launchFlags)
{
	Err error;

	error = RomVersionCompatible (ourMinVersion, launchFlags);
	if (error) return (error);

	switch (cmd)
		{
		case sysAppLaunchCmdNormalLaunch:
			error = AppStart();
			if (error) 
				return error;
				
			FrmGotoForm(MainForm);
			AppEventLoop();
			AppStop();
			break;

		default:
			break;

		}
	
	return errNone;
}


/***********************************************************************
 *
 * FUNCTION:    PilotMain
 *
 * DESCRIPTION: This is the main entry point for the application.
 *
 * PARAMETERS:  cmd - word value specifying the launch code. 
 *              cmdPB - pointer to a structure that is associated with the launch code. 
 *              launchFlags -  word value providing extra information about the launch.
 * RETURNED:    Result of launch
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
UInt32 PilotMain( UInt16 cmd, MemPtr cmdPBP, UInt16 launchFlags)
{
    return StarterPalmMain(cmd, cmdPBP, launchFlags);
}

