// Serial.java - Interface to the serial port for IR
//
// Note: only minimal IrDA implementation. (9600, fixed #BOFs)
//
// Copyright (c) 2000, The-Box Development
// Written by Jac Kersing <j.kersing@the-box.com>
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without 
// modification, are permitted provided that the following conditions are met:
// 
// Redistributions of source code must retain the above copyright notice, this 
// list of conditions and the following disclaimer.
// 
// Redistributions in binary form must reproduce the above copyright notice, 
// this list of conditions and the following disclaimer in the documentation 
// and/or other materials provided with the distribution.
//
// All advertising materials mentioning features or use of this software must 
// display the following acknowledgement: 
// 
// This product includes software developed by The-Box Development.
// 
// The name of The-Box Development may not be used to endorse or promote 
// products derived from this software without specific prior written 
// permission. 
//
// THIS SOFTWARE IS PROVIDED BY THE-BOX DEVELOPMENT ``AS IS'' AND ANY EXPRESS 
// OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
// NO EVENT SHALL THE-BOX DEVELOPMENT BE LIABLE FOR ANY DIRECT, INDIRECT, 
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, 
// OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
//


package nl.tbdev.IR;

import javax.comm.*;
import java.io.*;
import nl.tbdev.IR.Fcs;

public class Serial {
	public final byte XBOF = (byte) 0xff;
	public final byte BOF = (byte) 0xc0;
	public final byte EOF = (byte) 0xc1;
	public final byte CE = (byte) 0x7d;
	public final byte XVAL = (byte) 0x20;

	protected InputStream serIn = null;
	protected OutputStream serOut = null;
	protected SerialPort serialPort = null;
	private Fcs fcs = null;
	private byte[] lFrame;
	private Frame dFrame;
	private int offset;
	private int state;
	private byte[] data;
	private int pos;
	private int num;

	// constructor. open serial port
   public Serial() {
		fcs = new Fcs();
		dFrame = new Frame();
		lFrame = dFrame.data;
		offset = 0;
		state = GARBAGE;
		data = new byte[100];
   }

	private final int GARBAGE = 1;
	private final int SYNCING = 2;
	private final int XBOFSYNCING = 3;
	private final int READING = 4;
	private final int ESCREADING = 5;

	// read frame from the serial stream, processing specials
   public Frame readFrame() {
		byte d;
		int count = 0;
		boolean frameComplete = false;
		
		// continue until all of the frame has been read
		while (! frameComplete)
		{
			if (pos >= num) {
				pos = 0;
				try {
					num = serIn.read(data);
				} catch (Exception e) {
					System.out.println("Exception: "+e.toString());
				}
			}
			if (num > 0) {
				d = data[pos++];
				if (state == GARBAGE) {
					if (d == BOF) {
						state = SYNCING;
						continue;
					} else if ( d == XBOF ) {
						state = XBOFSYNCING;
						continue;
					}
				}
				if (state == XBOFSYNCING) {
					// should count the number of BOF(s) here?
					if (d == BOF) {
						offset = 0;
						state = READING;
						continue;
					}
				}
				if (state == SYNCING) {
					// should count the number of BOF(s) here?
					if (d != BOF) {
						offset = 0;
						state = READING;
					}
				}
				if (state == READING) {
					if (d == EOF) {
						frameComplete = true;
					} else if (d == CE) {
						state = ESCREADING;
					} else {
						lFrame[offset++] = d;
					}
				} else if (state == ESCREADING) {
					if (d == CE) {
						// abort should check for EOF
						offset = 0;
						state = GARBAGE;
					} else {
						lFrame[offset++] = (byte) (d ^ XVAL);
						state = READING;
					}
				}
			} else {
				// timeout on read. return to caller.
				return null;
			}
		}

		// for the next time around
		state = GARBAGE;

		// CRC check
		fcs.initFcs();
		fcs.calcFcs(lFrame,offset);
		if (!fcs.checkFcs()) {
			//System.out.println("CRC check on input failed!");
			// Could throw an exception here, but no result is as effective at
			// the moment.
			return null;
		}

		dFrame.length = offset - 2;
		//display("in",lFrame,offset);
		return dFrame;
	}
	
	// NOTE: Minimal implementation only! # XBOFs == default!!
	public void writeFrame(byte[] frame) {
		// create array for output frame. Worst case every byte needs
		// an escape...
		byte[] data = new byte[frame.length * 2 + 14];
		int i;
		int offset = 11;
		byte ch;

		for (i = 0; i < 10; i++) {
			data[i] = XBOF;
		}
		data[10] = BOF;
		for (i = 0; i < frame.length; i++) {
			ch = frame[i];
			if (ch == BOF || ch == EOF || ch == CE) {
				data[offset++] = CE;
				data[offset++] = (byte) (ch ^ XVAL);
			} else {
				data[offset++] = ch;
			}
		}
		// calculate and insert CRC here (LSB first)
		fcs.initFcs();
		fcs.calcFcs(frame,frame.length);
		short crc = fcs.getFcs();
		crc ^= 0xffff;
		byte out = (byte) (crc & 0xff);
		if (out == BOF || out == EOF || out == CE) {
			data[offset++] = CE;
			data[offset++] = (byte) (out ^ XVAL);
		} else {
			data[offset++] = out;
		}
		out = (byte) ((crc >> 8) & 0xff);
		if (out == BOF || out == EOF || out == CE) {
			data[offset++] = CE;
			data[offset++] = (byte) (out ^ XVAL);
		} else {
			data[offset++] = out;
		}
		data[offset++] = EOF;

		//display("out",data,offset);
		try {
			serOut.write(data,0,offset);
		} catch (Exception e) {
			System.out.println("Exception: "+e.toString());
		}
	}

	private void display(String direction, byte[] data, int len) {
		System.out.println("buffer["+direction+"]: ");
		for (int i = 0; i < len; i++) {
			System.out.print(Display.hex(data[i])+" ");
		}
		System.out.println("");
	}
}
// vi: ts=3 sw=3 ai
