// LmpStation.java: LMP Station Control State Machine.
//
// Copyright (c) 2000, The-Box Development
// Written by Jac Kersing <j.kersing@the-box.com>
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without 
// modification, are permitted provided that the following conditions are met:
// 
// Redistributions of source code must retain the above copyright notice, this 
// list of conditions and the following disclaimer.
// 
// Redistributions in binary form must reproduce the above copyright notice, 
// this list of conditions and the following disclaimer in the documentation 
// and/or other materials provided with the distribution.
//
// All advertising materials mentioning features or use of this software must 
// display the following acknowledgement: 
// 
// This product includes software developed by The-Box Development.
// 
// The name of The-Box Development may not be used to endorse or promote 
// products derived from this software without specific prior written 
// permission. 
//
// THIS SOFTWARE IS PROVIDED BY THE-BOX DEVELOPMENT ``AS IS'' AND ANY EXPRESS 
// OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
// NO EVENT SHALL THE-BOX DEVELOPMENT BE LIABLE FOR ANY DIRECT, INDIRECT, 
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, 
// OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
//

// This code is based upon:
// Infrared Data Association
// Minimal IrDA Protocol Implementation (IrDA Lite)
// version 1.0, November 7, 1996

package nl.tbdev.IR;
import nl.tbdev.IR.LapSecondary;
import java.util.Enumeration;

//
// N.B. This version is for a secondary only role. This means
// discovery and connection setup has not been implemented
//
public class LmpStation implements LapCallbackInterface {
	// define the states of the state machine
	public final static int	SDISC = 0x00;
	public final static int SSETUP = 0x01;
	public final static int DISCOVER = 0x02;
	public final static int SACTIVE = 0x03;

	// class variables
	private int state;
	private LapSecondary lap;
	private LmMux mux;
	private boolean disconnect = false;

	public LmpStation(Serial ser, LmMux mux) {
		lap = new LapSecondary(ser, this, 40);
		this.mux = mux;
		state = SDISC;
	}

	// call this to start the stack
	public void run() {
		while ( true ) {
			lap.run();
		}
	}
		
	public void disconnectRequest() {
		switch (state) {
			case SDISC:
				// nothing to be done
				break;
			case SACTIVE:
				// LM_LinkDisconnect.indication
				linkDisconnect();
				state = SDISC;
				break;
		}
		// do we want to do this??
		//lap.disconnectRequest();
	}
		
	public void linkConnectRequest() {
		// To-Be-Done, not for a secondary anyway
	}

	public void discoveryRequest() {
		// To-Be-Done, not for a secondary anyway
	}

	public void lapCallbackDisconnect() {
		// LM_LinkDisconnect.indication
		linkDisconnect();

		//System.out.println("Callback: DISCONNECT");
		state = SDISC;
	}

	public boolean lapCallbackConnect() {
		// LM_LinkConnect.indication
		Enumeration stations = mux.getRegistered();
		while (stations.hasMoreElements()) {
			LmpCallbackInterface station = (LmpCallbackInterface) 
															stations.nextElement();
			station.lmpCallbackConnect();
		}

		//System.out.println("Callback: CONNECT");
		return true;
	}

	public void lapCallbackData(byte[] data) {
		LmpFrame frame = new LmpFrame(data);

		// is this LM-Access?
		if (frame.isControl() && (frame.getOpcode() == LmpFrame.LMPACCESSMODE)) {
			lap.disconnectRequest();
			// LM_LinkDisconnect.indication
			linkDisconnect();
			state = SDISC;
			return;
		}

		// DLSAP in set of connections
		LmpCallbackInterface station = mux.getConnection(frame.getDlsap());
		if (station != null) {
			// valid destination, call callback with data
			station.lmpCallbackData(this,frame);
			return;
		}

		// everything else fails...
		state = SDISC;
		lap.disconnectRequest();
		// LM_LinkDisconnect.indication
		linkDisconnect();

		// System.out.println("Callback: DATA");
		//for (int i = 2; i < data.length; i++) {
		//	System.out.print(" "+Display.hex(data[i]));
		//}
		//System.out.println("");
	}

	private void linkDisconnect() {
		Enumeration stations = mux.getRegistered();

		while (stations.hasMoreElements()) {
			LmpCallbackInterface station = (LmpCallbackInterface) 
															stations.nextElement();
			station.lmpCallbackDisconnect();
		}
	}
			
	public void dataRequest(LmpFrame frame) {
		lap.dataRequest(frame.getFrame());
	}
}

// vi: ts=3 sw=3 ai
