// LapFrame: Class for manipulating a LMP frame
//
// Copyright (c) 2000, The-Box Development
// Written by Jac Kersing <j.kersing@the-box.com>
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without 
// modification, are permitted provided that the following conditions are met:
// 
// Redistributions of source code must retain the above copyright notice, this 
// list of conditions and the following disclaimer.
// 
// Redistributions in binary form must reproduce the above copyright notice, 
// this list of conditions and the following disclaimer in the documentation 
// and/or other materials provided with the distribution.
//
// All advertising materials mentioning features or use of this software must 
// display the following acknowledgement: 
// 
// This product includes software developed by The-Box Development.
// 
// The name of The-Box Development may not be used to endorse or promote 
// products derived from this software without specific prior written 
// permission. 
//
// THIS SOFTWARE IS PROVIDED BY THE-BOX DEVELOPMENT ``AS IS'' AND ANY EXPRESS 
// OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
// NO EVENT SHALL THE-BOX DEVELOPMENT BE LIABLE FOR ANY DIRECT, INDIRECT, 
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, 
// OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
//


package nl.tbdev.IR;

// N.B. The frame is supposed to have a LAP header at all times!
// (so all offsets are off by 2)
public class LmpFrame {
	byte[] frame;

	// this constructor only stores a pointer to the original data
	// keep this in mind (do not modify the data or something unpredictable
	// might happen!
	public LmpFrame(byte[] iframe) {
		super();
		frame = iframe;
	}

	public LmpFrame(int size) {
		super();
		frame = new byte[size + 2];
	}

	// is this a control frame?
	public boolean isControl() {
		return (frame[2] & 0x80) == 0x80;
	}

	public boolean isABit() {
		return (frame[4] & 0x80) == 0x80;
	}

	public void setABit() {
		byte tmp = (byte) (frame[4] & 0x7f);
		tmp |= 0x80;
		frame[4] = tmp;
	}  

	public byte getDlsap() {
		return (byte) (frame[2] & 0x7f);
	}
	
	public void setDlsap(byte lsap) {
		byte tmp = (byte) (frame[2] & 0x80);
		tmp |= lsap;
		frame[2] = tmp;
	}
	
	public byte getSlsap() {
		return (byte) (frame[3] & 0x7f);
	}
	
	public void setSlsap(byte lsap) {
		byte tmp = (byte) (frame[3] & 0x80);
		tmp |= lsap;
		frame[3] = tmp;
	}
	
	public static final byte LMPCONNECT = (byte) 0x01;
	public static final byte LMPDISCONNECT = (byte) 0x02;
	public static final byte LMPACCESSMODE = (byte) 0x03;

	public byte getOpcode() {
		return (byte) (frame[4] & 0x7f);
	}

	public void setOpcode(byte opcode) {
		byte tmp = frame[4];
		tmp &= (byte) 0x80;
		tmp |= (byte) (opcode & 0x7f);
		frame[4] = tmp;
	}

	public byte getParameter(int nr) {
		return frame[4 + nr];
	}

	public void setParameter(int nr, int value) {
		frame[4 + nr] = (byte) value;
	}

	public byte[] getFrame() {
		return frame;
	}

	public byte[] getData() {
		byte[] data = new byte[frame.length - 4];
		for (int i = 4; i < frame.length; i++) {
			data[i-4] = frame[i];
		}
		return data;
	}

	public void putData(byte[] data) {
		for (int i = 0; i < data.length; i++) {
			frame[4+i] = data[i];
		}
	}
}

// vi: ts=3 sw=3 ai
