// LmpConnection.java: LMP Connection Control State Machine.
//
// Copyright (c) 2000, The-Box Development
// Written by Jac Kersing <j.kersing@the-box.com>
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without 
// modification, are permitted provided that the following conditions are met:
// 
// Redistributions of source code must retain the above copyright notice, this 
// list of conditions and the following disclaimer.
// 
// Redistributions in binary form must reproduce the above copyright notice, 
// this list of conditions and the following disclaimer in the documentation 
// and/or other materials provided with the distribution.
//
// All advertising materials mentioning features or use of this software must 
// display the following acknowledgement: 
// 
// This product includes software developed by The-Box Development.
// 
// The name of The-Box Development may not be used to endorse or promote 
// products derived from this software without specific prior written 
// permission. 
//
// THIS SOFTWARE IS PROVIDED BY THE-BOX DEVELOPMENT ``AS IS'' AND ANY EXPRESS 
// OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
// NO EVENT SHALL THE-BOX DEVELOPMENT BE LIABLE FOR ANY DIRECT, INDIRECT, 
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, 
// OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
//

// This code is based upon:
// Infrared Data Association
// Minimal IrDA Protocol Implementation (IrDA Lite)
// version 1.0, November 7, 1996

package nl.tbdev.IR;
import nl.tbdev.IR.LapSecondary;
import java.util.Enumeration;

//
// N.B. This version is for a secondary only role. This means
// discovery and connection setup has not been implemented
//
public class LmpConnection implements LmpCallbackInterface {
	// define the states of the state machine
	public final static int	NOT_READY = 0x00;
	public final static int	READY = 0x01;
	public final static int CSETUP = 0x02;
	public final static int INCOMING = 0x03;
	public final static int ACTIVE = 0x04;

	// class variables
	private int state;
	ClientCallbackInterface client = null;
	private LmpStation station;

	public LmpConnection(ClientCallbackInterface client) {
		super();
		state = NOT_READY;
		this.client = client;
	}
		
	public void lmpCallbackConnect() {
		if (state == NOT_READY) {
			state = READY;
		}
	}

	public void lmpCallbackData(LmpStation station, LmpFrame frame) {
		this.station = station;
		switch (state) {
			case READY:
					readyData(frame);
					break;
			case CSETUP:
					csetupData(frame);
					break;
			case ACTIVE:
					activeData(frame);
					break;
		}
	}

	public void lmpCallbackDisconnect() {
		// LM_Disconnect.indication
		client.clientCallbackDisconnect();

		state = NOT_READY;
	}

	private void readyData(LmpFrame frame) {
		if (frame.isControl() && (frame.getOpcode() == LmpFrame.LMPCONNECT)) {
			if (frame.isABit()) {
				station.disconnectRequest();
				client.clientCallbackDisconnect();
				state = NOT_READY;
				return;
			}
			// LM_connect.indication
			if (client.clientCallbackConnect()) {
				byte[] arg = { (byte) 0x00 };
				LmpControlFrame rf = new LmpControlFrame(
												frame.getSlsap(),
												frame.getDlsap(),
												LmpFrame.LMPCONNECT,
												true,
												arg);
				station.dataRequest(rf);
				state = ACTIVE;
			} else {
				state = NOT_READY;
				return;
			}
		}
		if (!frame.isControl()) {
			station.disconnectRequest();
			client.clientCallbackDisconnect();
			state = NOT_READY;
			return;
		}
	}

	private void csetupData(LmpFrame frame) {
		if (frame.isControl() && frame.isABit() && 
			(frame.getOpcode() == LmpFrame.LMPCONNECT) ) {
			// LM_connect.confirm
			client.clientCallbackConfirm();
			state = ACTIVE;
			return;
		}
		if (frame.isControl() && 
			( (frame.getOpcode() == LmpFrame.LMPDISCONNECT) ||
			  (frame.getOpcode() == LmpFrame.LMPCONNECT) ) ) {
			// LM_Disconnect.indication
			client.clientCallbackDisconnect();
			state = READY;
			return;
		}
	}

	private void activeData(LmpFrame frame) {
		// regular data --> pass to upper layer
		if (!frame.isControl()) {
			// LM_Data.indication
			byte[] rframe = frame.getFrame();
			byte[] data = new byte[rframe.length - 4];
			for (int i = 4; i < rframe.length; i++) {
				data[i-4] = rframe[i];
			}
			client.clientCallbackData(this, frame.getSlsap(), data);
			return;
		}
		// LM Connect --> disconnect
		if (frame.isControl() && !frame.isABit() && 
			(frame.getOpcode() == LmpFrame.LMPCONNECT) ) {
			station.disconnectRequest();
			client.clientCallbackDisconnect();
			state = NOT_READY;
			return;
		}
		if (frame.isControl() &&
			(frame.getOpcode() == LmpFrame.LMPDISCONNECT) ) {
			// LM_Disconnect.indication
			client.clientCallbackDisconnect();
			state = READY;
			return;
      }
	}

	public void dataRequest(byte lLsap, byte rLsap, byte[] data) {
		LmpFrame frame = new LmpFrame(2 + data.length);
		frame.setDlsap(rLsap);
		frame.setSlsap(lLsap);
		frame.putData(data);
		station.dataRequest(frame);
	}
}

// vi: ts=3 sw=3 ai
