// LapSnrmFrame : Implements a SNRM type of LAP frame.
//
// Copyright (c) 2000, The-Box Development
// Written by Jac Kersing <j.kersing@the-box.com>
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without 
// modification, are permitted provided that the following conditions are met:
// 
// Redistributions of source code must retain the above copyright notice, this 
// list of conditions and the following disclaimer.
// 
// Redistributions in binary form must reproduce the above copyright notice, 
// this list of conditions and the following disclaimer in the documentation 
// and/or other materials provided with the distribution.
//
// All advertising materials mentioning features or use of this software must 
// display the following acknowledgement: 
// 
// This product includes software developed by The-Box Development.
// 
// The name of The-Box Development may not be used to endorse or promote 
// products derived from this software without specific prior written 
// permission. 
//
// THIS SOFTWARE IS PROVIDED BY THE-BOX DEVELOPMENT ``AS IS'' AND ANY EXPRESS 
// OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
// NO EVENT SHALL THE-BOX DEVELOPMENT BE LIABLE FOR ANY DIRECT, INDIRECT, 
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, 
// OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
//


package nl.tbdev.IR;
import nl.tbdev.IR.LapFrame;

public class LapSnrmFrame extends LapFrame {
	public LapSnrmFrame(byte[] frame) {
		super(frame);
	}

	public LapSnrmFrame(LapFrame frame) {
		super(frame.getFrame());
	}

	public byte[] getOrigAddressByte() {
		byte[] addr = new byte[4];
		addr[0] = frame[2];
		addr[1] = frame[3];
		addr[2] = frame[4];
		addr[3] = frame[5];
		return addr;
	}

	public int getOrigAddressInt() {
		return ((frame[2] << 24) & 0xff000000) | ((frame[3] << 16) & 0xff0000) | 
				 ((frame[4] << 8) & 0xff00) | (frame[5] & 0xff);
	}

	public byte[] getDestAddressByte() {
		byte[] addr = new byte[4];
		addr[0] = frame[6];
		addr[1] = frame[7];
		addr[2] = frame[8];
		addr[3] = frame[9];
		return addr;
	}

	public int getDestAddressInt() {
		return ((frame[6] << 24) & 0xff000000) | ((frame[7] << 16) & 0xff0000) | 
				 ((frame[8] << 8) & 0xff00) | (frame[9] & 0xff);
	}

	public boolean resetFrame() {
		return (frame.length <= 10);
	}

	public byte getConnectAddress() {
		return (byte) ((frame[10] >>> 1) & 0x7f);
	}

	// we don't need these for a minimal IrDA implementation
	public byte[] getOptions() {
		byte[] opts = new byte[frame.length - 10];
		for (int i = 11; i < frame.length; i++) {
			opts[i - 11] = frame[i];
		}
		return opts;
	}

	private static final byte PI_BAUD = (byte) 0x01;
	private static final byte PI_MAXTURN = (byte) 0x82;
	private static final byte PI_DATASIZE = (byte) 0x83;
	private static final byte PI_WINSIZE = (byte) 0x84;
	private static final byte PI_ADDBOF = (byte) 0x85;
	private static final byte PI_MINTURN = (byte) 0x86;
	private static final byte PI_LINKDIS = (byte) 0x08;

	// return options in String form, suitable for printing
	public String stringOptions() {
		String opts = "";
		int i = 11;
		while ( i < frame.length ) {
			switch (frame[i]) {
				case PI_BAUD:
					opts += "connection speed: ";
					if (frame[i+1] == 2) {
						if((frame[i+2] & 0x01) != 0) {
							opts += "4000000 bps ";
						}
						i++;
					}
					if ((frame[i+2] & 0x80) != 0) {
						opts += "1152000 bps ";
					}
					if ((frame[i+2] & 0x40) != 0) {
						opts += "576000 bps ";
					}
					if ((frame[i+2] & 0x20) != 0) {
						opts += "115200 bps ";
					}
					if ((frame[i+2] & 0x10) != 0) {
						opts += "57600 bps ";
					}
					if ((frame[i+2] & 0x08) != 0) {
						opts += "38400 bps ";
					}
					if ((frame[i+2] & 0x04) != 0) {
						opts += "19200 bps ";
					}
					if ((frame[i+2] & 0x02) != 0) {
						opts += "9600 bps ";
					}
					if ((frame[i+2] & 0x01) != 0) {
						opts += "2400 bps ";
					}
					break;
				case PI_MAXTURN:
					opts += "max turn around time: ";
					if ((frame[i+2] & 0x08) != 0) {
						opts += "50 ms ";
					}
					if ((frame[i+2] & 0x04) != 0) {
						opts += "100 ms ";
					}
					if ((frame[i+2] & 0x02) != 0) {
						opts += "250 ms ";
					}
					if ((frame[i+2] & 0x01) != 0) {
						opts += "500 ms ";
					}
					break;
				case PI_DATASIZE:
					opts += "data size: ";
					if ((frame[i+2] & 0x20) != 0) {
						opts += "2048 bytes ";
					}
					if ((frame[i+2] & 0x10) != 0) {
						opts += "1024 bytes ";
					}
					if ((frame[i+2] & 0x08) != 0) {
						opts += "512 bytes ";
					}
					if ((frame[i+2] & 0x04) != 0) {
						opts += "256 bytes ";
					}
					if ((frame[i+2] & 0x02) != 0) {
						opts += "128 bytes ";
					}
					if ((frame[i+2] & 0x01) != 0) {
						opts += "64 bytes ";
					}
					break;
				case PI_WINSIZE:
					opts += "window size: ";
					if ((frame[i+2] & 0x40) != 0) {
						opts += "7 frame window ";
					}
					if ((frame[i+2] & 0x20) != 0) {
						opts += "6 frame window ";
					}
					if ((frame[i+2] & 0x10) != 0) {
						opts += "5 frame window ";
					}
					if ((frame[i+2] & 0x08) != 0) {
						opts += "4 frame window ";
					}
					if ((frame[i+2] & 0x04) != 0) {
						opts += "3 frame window ";
					}
					if ((frame[i+2] & 0x02) != 0) {
						opts += "2 frame window ";
					}
					if ((frame[i+2] & 0x01) != 0) {
						opts += "1 frame window ";
					}
					break;
				case PI_ADDBOF:
					opts += "additional BOFs (at 115200 bps): ";
					if ((frame[i+2] & 0x80) != 0) {
						opts += "0 ";
					}
					if ((frame[i+2] & 0x40) != 0) {
						opts += "1 ";
					}
					if ((frame[i+2] & 0x20) != 0) {
						opts += "2 ";
					}
					if ((frame[i+2] & 0x10) != 0) {
						opts += "3 ";
					}
					if ((frame[i+2] & 0x08) != 0) {
						opts += "5 ";
					}
					if ((frame[i+2] & 0x04) != 0) {
						opts += "12 ";
					}
					if ((frame[i+2] & 0x02) != 0) {
						opts += "24 ";
					}
					if ((frame[i+2] & 0x01) != 0) {
						opts += "48 ";
					}
					break;
				case PI_MINTURN:
					opts += "minimum turn around time:  ";
					if ((frame[i+2] & 0x80) != 0) {
						opts += "0 ms ";
					}
					if ((frame[i+2] & 0x40) != 0) {
						opts += "0.01 ms ";
					}
					if ((frame[i+2] & 0x20) != 0) {
						opts += "0.05 ms ";
					}
					if ((frame[i+2] & 0x10) != 0) {
						opts += "0.1 ms ";
					}
					if ((frame[i+2] & 0x08) != 0) {
						opts += "0.5 ms ";
					}
					if ((frame[i+2] & 0x04) != 0) {
						opts += "1 ms ";
					}
					if ((frame[i+2] & 0x02) != 0) {
						opts += "5 ms ";
					}
					if ((frame[i+2] & 0x01) != 0) {
						opts += "10 ms ";
					}
					break;
				case PI_LINKDIS:
					opts += "link disconnect (threshold) time: ";
					if ((frame[i+2] & 0x80) != 0) {
						opts += "40 secs (3 secs) ";
					}
					if ((frame[i+2] & 0x40) != 0) {
						opts += "30 secs (3 secs) ";
					}
					if ((frame[i+2] & 0x20) != 0) {
						opts += "25 secs (3 secs) ";
					}
					if ((frame[i+2] & 0x10) != 0) {
						opts += "20 secs (3 secs) ";
					}
					if ((frame[i+2] & 0x08) != 0) {
						opts += "16 secs (3 secs) ";
					}
					if ((frame[i+2] & 0x04) != 0) {
						opts += "12 secs (3 secs) ";
					}
					if ((frame[i+2] & 0x02) != 0) {
						opts += "8 secs (3 secs) ";
					}
					if ((frame[i+2] & 0x01) != 0) {
						opts += "4 secs (0 secs) ";
					}
					break;
				default:
					opts += "unknown parameter " + frame[i] + " ";
			}
			i += 3;
		}
		return opts;
	}
}

// vi: sw=3 ts=3 ai
