// LapSecondary: state machine for secondary only IrDA LAP role
//
// Copyright (c) 2000, The-Box Development
// Written by Jac Kersing <j.kersing@the-box.com>
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without 
// modification, are permitted provided that the following conditions are met:
// 
// Redistributions of source code must retain the above copyright notice, this 
// list of conditions and the following disclaimer.
// 
// Redistributions in binary form must reproduce the above copyright notice, 
// this list of conditions and the following disclaimer in the documentation 
// and/or other materials provided with the distribution.
//
// All advertising materials mentioning features or use of this software must 
// display the following acknowledgement: 
// 
// This product includes software developed by The-Box Development.
// 
// The name of The-Box Development may not be used to endorse or promote 
// products derived from this software without specific prior written 
// permission. 
//
// THIS SOFTWARE IS PROVIDED BY THE-BOX DEVELOPMENT ``AS IS'' AND ANY EXPRESS 
// OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
// NO EVENT SHALL THE-BOX DEVELOPMENT BE LIABLE FOR ANY DIRECT, INDIRECT, 
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, 
// OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
//

// This code is based upon:
// Infrared Data Association
// Minimal IrDA Protocol Implementation (IrDA Lite)
// version 1.0, November 7, 1996

package nl.tbdev.IR;

import nl.tbdev.IR.Serial;
import nl.tbdev.IR.LapFrame;
import java.util.Vector;

// In the ideal world this class would be a thread by itself.
// Not now...
public class LapSecondary {
	// define the states of the state machine
	public final static int	OFFLINE = 0x00;
	public final static int NDM = 0x01;
	public final static int CONN = 0x02;
	public final static int NRMS = 0x03;
	public final static int SCLOSE = 0x04;

	// Class variables
	private int state = OFFLINE;
	private Serial serial = null;
	private int timeout = 40;
	private java.util.Random generator = null;
	private byte old_s = 127;
	private int Ns, Vr, Vs, As;
	private int NA;
	private LapFrame frame = null;
	private int slot = 0;
	private boolean frameSent = true;
	private boolean remoteBusy = false;
	private boolean localBusy = false;
	private boolean unackedFrame = false;
	private byte[] data = null;
	private long expireTime = 0;
	private boolean disconnect = false;
	private Vector outQueue;

	// Callbacks for Connect.Indication, Data.Indication and 
	// Disconnect.Indication
	LapCallbackInterface callback = null;

	// Save parameters of connection
	private byte[] dest = null;
	private byte ca = (byte) 0;

	public LapSecondary(Serial iSerial, LapCallbackInterface cb, int iTimeout) {
		super();
		serial = iSerial;
		timeout = iTimeout;
		callback = cb;
		disconnect = false;
		generator = new java.util.Random(System.currentTimeMillis());
		outQueue = new Vector();
	}

	public void disconnectRequest() {
		disconnect = true;
	}

	// data to be send (includes frame header!)
	public void dataRequest(byte[] data) {
		outQueue.addElement(data);
	}
		
	public void run() {
		state = OFFLINE;

		// enter the state machine...
		while ( true ) {
			switch (state) {
				case OFFLINE:
						//System.out.println("State: OFFLINE");
						handleOffline();
						break;
				case NDM:
						//System.out.println("State: NDM");
						handleNdm();
						break;
				case CONN:
						//System.out.println("State: CONN");
						handleCon();
						break;
				case NRMS:
						//System.out.println("State: NRMS");
						handleNrms();
						break;
				case SCLOSE:
						//System.out.println("State: SCLOSE");
						handleSclose();
						break;
			}
		}
	}

	// we're currently OFFLINE. setup and go to NDM
	private void handleOffline() {
		NA = generator.nextInt();
		//System.out.println("Starting with NA: "+Display.hex(NA));
		old_s = 127;
		state = NDM;
	}

	private void handleNdm() {
		// ignore link-shutdown for now, if user disables station software
		// isn't likely to run anyway (at least not for long)
		
		// read a frame
		Frame rframe = serial.readFrame();
		if (rframe ==  null) {
			// no frame received, CRC failed or (link) timeout
			//System.out.println("Got empty frame");
			return;
		}

		// get handle on data
		byte[] fr = rframe.data;

		// encapsulate it
		//frame = LapFrame.newFrame(rframe);

		// start checking the conditions
//		if (frame.isUnnumbered() && frame.isControl() &&
//				frame.unnumberedControl() == LapFrame.XIDc &&
//				frame.isEndDiscovery() ) {
		if ( ((fr[1] & 0x03) == 0x03) &&				// isUnnumbered
		     ((fr[0] & 0x01) == 0x01) &&				// isControl
			  ((fr[1] & 0xef) == LapFrame.XIDc) ) { // is XIDc

			// got a discovery frame, is it EndDiscovery?
			if ( fr[12] == 0xff ) {						// isEndDiscovery
				//System.out.println("Got End-Discovery-XID-Cmd");
				old_s = 127;
				return;
			}

			// ! isEndDiscovery -> normal discovery frame
//		if (frame.isUnnumbered() && frame.isControl() &&
//				frame.unnumberedControl() == LapFrame.XIDc) {
//			int s = frame.getSlot();
//			int S = frame.getMaxSlot()-1;
			int s = fr[12];
			int S = fr[11] & 0x03;
			S = ( S == 0 ) ? 0 : ( S == 1 ) ? 5 : 7;
			//System.out.println("Got Discovery-XID-Cmd "+(S+1)+" "+s);

			if ( s <= old_s ) {
				slot = generator.nextInt() & 0x1f;
				if (S-s != 0) {
					slot %= (S-s);
				} else {
					slot = 0;
				}
				slot += s;
				//System.out.println("Randomizer, frame slot="+slot);
				frameSent = false;
			}
			old_s = (byte) s;
//			if ( ( s >= slot ) && ( ! frame.isEndDiscovery() ) && ( !frameSent ) ) {
			if ( ( s >= slot ) && ( !frameSent ) ) {
//				if ( frame.isAddressConflict() ) {
				if ( (fr[11] & 0x04 ) == 0x04 ) {
					NA = generator.nextInt();
					//System.out.println("Generated new NA: "+Display.hex(NA));
				}

				// create a reply frame
				byte[] info = { 			(byte) 0x82, (byte) 0x20, (byte) 0x00, 
												(byte) 'J', (byte) 'a', (byte) 'v',
												(byte) 'a', (byte) 'I', (byte) 'r',
												(byte) 'D', (byte) 'A' };
				LapXIDFrame reply = new LapXIDFrame(true, info.length);
//				LapXIDFrame org = (LapXIDFrame) frame;
				reply.setOrigAddress(NA);
//				reply.setDestAddress(org.getOrigAddressByte());
				byte[] addr = new byte[4];
				for (int i=0; i<4; i++) {
					addr[i] = fr[i+3];
				}
				reply.setDestAddress(addr);
				reply.setMaxSlot(S+1);
//				reply.setAddressConflict(frame.isAddressConflict());
				reply.setAddressConflict((fr[11] & 0x04 ) == 0x04);
//				reply.setSlot(frame.getSlot());
				reply.setSlot(fr[12]);
				reply.setInfo(info);

				// send the reply
				serial.writeFrame(reply.getFrame());
				//System.out.println("Reply sent");
				frameSent = true;
			}
			return;
		}
//		if ( frame.isUnnumbered() && frame.isPoll() &&
//			  (frame.unnumberedControl() == LapFrame.SNRMc) ) {
		if ( ((fr[1]& 0x03) == 0x03) &&								// isUnnumbered
			  ((fr[1] & 0x10) == 0x10) &&								// isPoll
			  ((fr[1] & 0xef) == (LapFrame.SNRMc&0xff)) ) {		// is SNRMc
			// save connection address and destination address
			//System.out.println("Recv u:snrm:cmd:P");
//			LapSnrmFrame f = (LapSnrmFrame) frame;
			LapSnrmFrame f = (LapSnrmFrame) LapFrame.newFrame(rframe);
			dest = f.getOrigAddressByte();
			ca = f.getConnectAddress();

			// check if this was targetted at us
			if (NA == f.getDestAddressInt()) {
				state = CONN;
			} else {
				//System.out.println("Not to us");
			}
			//System.out.println("From: "+Display.hex(f.getOrigAddressInt()));
			//System.out.println("To  : "+Display.hex(f.getDestAddressInt()));
			//System.out.println("Addr: "+Display.hex(f.getConnectAddress()));
			//System.out.println("OPTIONS:");
			//System.out.println(f.stringOptions());
		}
	}

	private void handleCon() {
		// call callback to check if connection wanted.
		if (! callback.lapCallbackConnect()) {
			state = NDM;
		} else {
		//System.out.println("Accept connection");
			// negotiate-connection-parameters (nothing to be done)

			// Send u:ua:rsp:F:Connection-Parameters
			LapUaFrame rf = new LapUaFrame(ca, true);
			rf.setDestAddress(dest);
			rf.setOrigAddress(NA);
			rf.setFinal();
			rf.setDefaultParameters();
			serial.writeFrame(rf.getFrame());
			//System.out.println("Sent options: "+rf.stringOptions());
			
			// Apply-Connection-Parameters (nothing to be done, are set already)

			// Initialize-Connection-State
			remoteBusy = localBusy = false;
			Vr = Vs = 0;
			As = 7;
			unackedFrame = false;
			data = null;

			// Start-WD-timer for quick disconnects
			StartWdTimer(2);

			state = NRMS;
		}
	}

	private void handleNrms() {
		// Data-Request : to be handled elsewhere

		// Disconnect-Request
		if (disconnect) {
			state = SCLOSE;
			disconnect = false;
		}

		// read a frame
		Frame rframe = serial.readFrame();
		if (rframe ==  null) {
			// no frame received, CRC failed or (link) timeout
			//System.out.println("Got empty frame");
			if (System.currentTimeMillis() > expireTime) {
				// watchdog time expired!
				old_s = 127;
				// Disconnect indication
				callback.lapCallbackDisconnect();
				state = NDM;
			}
			return;
		}

		// encapsulate it
		frame = LapFrame.newFrame(rframe);

		// check the connection address to see if this is our connection
		if (ca != frame.getAddress()) {
			// darn, something is wrong. this is not for us (now)
			return;
		}

		// start checking the conditions
		// Recv I:cmd:Ns:Nr:P && Invalid-Nr (first so it doesn't bother us
		// later on)
		if (frame.isInformation() && frame.isControl() &&
				frame.isPoll() && 
				invalidNr( ((LapIFrame) frame).getNr(), Vs, unackedFrame, As)) {
			sendURdRspF();
			state = SCLOSE;
			return;
		}

		// Recv I:cmd:Ns:Nr:P
		if (frame.isInformation() && frame.isControl() && frame.isPoll()) {
			if ( ! localBusy ) {
				Ns = ((LapIFrame) frame).getNs();
				if ( Ns == Vr ) {
					Vr = (Vr + 1) % 8;
					// Data indication! pass data (including header) to upper layer
					callback.lapCallbackData(frame.getFrame());
					SendNrmResponse(((LapIFrame)frame).getNr());
				} else {
					SendSResponse();
				}
			} else {
				SendNrmResponse(((LapSFrame)frame).getNr());
			}
			StartWdTimer(timeout);
			return;
		}

		// Recv s:x:cmd:Nr:P && Invalid-Nr
		// check this first
		if (frame.isSupervisory() && frame.isControl() &&
				frame.isPoll() && 
				invalidNr( ((LapSFrame)frame).getNr(), Vs, unackedFrame, As)) {
			sendURdRspF();
			state = SCLOSE;
			return;
		}

		// Recv s:rr:cmd:Nr:P
		if (frame.isSupervisory() && frame.isControl() &&
				frame.supervisoryControl() == LapFrame.RR) {
			//System.out.println("Got Receive Ready");
			remoteBusy = false;
			SendNrmResponse(((LapSFrame)frame).getNr());
			StartWdTimer(timeout);
			return;
		}

		// Recv s:rnr:cmd:Nr:P
		if (frame.isSupervisory() && frame.isControl() &&
				frame.supervisoryControl() == LapFrame.RNR) {
			//System.out.println("Got Receive Not Ready");
			remoteBusy = true;
			SendNrmResponse(((LapSFrame)frame).getNr());
			StartWdTimer(timeout);
			return;
		}

		// Recv s:rej:cmd:Nr:P ||
		// Recv s:srej:cmd:Nr:P
		if (frame.isSupervisory() && frame.isControl() &&
				( (frame.supervisoryControl() == LapFrame.REJ) ||
				  (frame.supervisoryControl() == LapFrame.SREJ) ) ) {
			//System.out.println("Got Reject ");
			SendNrmResponse(((LapSFrame)frame).getNr());
			StartWdTimer(timeout);
			return;
		}

		// Recv u:snrm:cmd:P
      if ( frame.isUnnumbered() && frame.isPoll() &&
			  (frame.unnumberedControl() == LapFrame.SNRMc) ) {
			//System.out.println("Got snrm frame");
			sendURdRspF();
			state = SCLOSE;
			return;
		}

		// Recv u:disc:cmd:P
		if ( frame.isUnnumbered() && frame.isPoll() &&
			  (frame.unnumberedControl() == LapFrame.DISCc) ) {
			//System.out.println("Got disconnect frame");
			LapUaFrame rf = new LapUaFrame(ca, false);
			rf.setFinal();
			rf.setOrigAddress(NA);
			rf.setDestAddress(dest);
			serial.writeFrame(rf.getFrame());
			// Apply-Default-Connection-Parameters
			old_s = 127;
			state = NDM;

			// Disconnect-Indication
			callback.lapCallbackDisconnect();
			return;
		}

		// Recv x:x:cmd:P
		if (frame.isControl() && frame.isPoll()) {
			SendSResponse();
			return;
		}

		// Recv x:x:x:x
		// do nothing...
	}

	private void handleSclose() {
		// read a frame
		Frame rframe = serial.readFrame();
		if (rframe ==  null) {
			// no frame received, CRC failed or (link) timeout
			//System.out.println("Got empty frame");
			if (System.currentTimeMillis() > expireTime) {
				// watchdog time expired!
				old_s = 127;
				// Disconnect indication
				callback.lapCallbackDisconnect();
				state = NDM;
			}
			return;
		}

		// encapsulate it
		frame = LapFrame.newFrame(rframe);

		// check the connection address to see if this is our connection
		if (ca != frame.getAddress()) {
			// darn, something is wrong. this is not for us (now)
			return;
		}

		// Recv u:disc:cmd:P
		if ( frame.isUnnumbered() && frame.isPoll() &&
			  (frame.unnumberedControl() == LapFrame.DISCc) ) {
			//System.out.println("Got disconnect frame");
			LapUaFrame rf = new LapUaFrame(ca, false);
			rf.setFinal();
			rf.setOrigAddress(NA);
			rf.setDestAddress(dest);
			serial.writeFrame(rf.getFrame());
			// Apply-Default-Connection-Parameters
			old_s = 127;
			state = NDM;
			// Disconnect-Indication
			callback.lapCallbackDisconnect();
			return;
		}

		// Recv x:x:cmd:P
		if (frame.isControl() && frame.isPoll()) {
			sendURdRspF();
			return;
		}
	}

	// set timeout value
	private void StartWdTimer(int timeout) {
		expireTime = System.currentTimeMillis() + (1000 * timeout);
	}

	private boolean invalidNr(int nr, int vs, boolean unackedframe, int as) {
		if ( (nr==vs) || (unackedframe && (nr==as)) ) {
			return false;
		}
		return true;
	}

	private void SendNrmResponse(int Nr) {
		//System.out.println("SendNrmResponse("+Nr+")");
		if (! remoteBusy) {
			if (Nr == As) {
				SendIRsp(As,Vr,data);
				unackedFrame = false;
			} else if ((outQueue.size() != 0) && (Nr == Vs)) {
				// get data in data
				data = (byte []) outQueue.firstElement();
				outQueue.removeElement(data);
				SendIRsp(Vs,Vr,data);
				unackedFrame = true;
				As = Vs;
				Vs = (Vs + 1) % 8;
			} else {
				SendSResponse();
			}
		} else {
			SendSResponse();
		}
	}

	private void SendIRsp(int As, int Vr, byte[] data) {
		LapIFrame rf = new LapIFrame(data);
		rf.setAddr(ca);
		rf.setNrNs(Vr,As);
		if (data != null) {
			rf.copyData(data);
		}
		serial.writeFrame(rf.getFrame());
	}

	private void SendSResponse() {
		//System.out.println("SendSResponse()");
		LapSFrame rf = new LapSFrame(ca, Vr);
		rf.setFinal();
		rf.setType( localBusy ? LapFrame.RNR : LapFrame.RR );
		serial.writeFrame(rf.getFrame());
	}

	private void sendURdRspF() {
		//System.out.println("Send RequestDisconnect");
		LapRdFrame rf = new LapRdFrame(ca);
		rf.setFinal();
		serial.writeFrame(rf.getFrame());
	}
}

// vi: ts=3 sw=3 ai
