// LapFrame: Class for manipulating a LAP frame.
//
// Copyright (c) 2000, The-Box Development
// Written by Jac Kersing <j.kersing@the-box.com>
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without 
// modification, are permitted provided that the following conditions are met:
// 
// Redistributions of source code must retain the above copyright notice, this 
// list of conditions and the following disclaimer.
// 
// Redistributions in binary form must reproduce the above copyright notice, 
// this list of conditions and the following disclaimer in the documentation 
// and/or other materials provided with the distribution.
//
// All advertising materials mentioning features or use of this software must 
// display the following acknowledgement: 
// 
// This product includes software developed by The-Box Development.
// 
// The name of The-Box Development may not be used to endorse or promote 
// products derived from this software without specific prior written 
// permission. 
//
// THIS SOFTWARE IS PROVIDED BY THE-BOX DEVELOPMENT ``AS IS'' AND ANY EXPRESS 
// OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
// NO EVENT SHALL THE-BOX DEVELOPMENT BE LIABLE FOR ANY DIRECT, INDIRECT, 
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, 
// OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
//


package nl.tbdev.IR;

import nl.tbdev.IR.LapSFrame;
import nl.tbdev.IR.LapSnrmFrame;
import nl.tbdev.IR.LapXIDFrame;

public class LapFrame {
	byte[] frame;

	// this constructor only stores a pointer to the original data
	// keep this in mind (do not modify the data or something unpredictable
	// might happen!
	public LapFrame(byte[] iframe) {
		super();
		frame = iframe;
	}

	public LapFrame(int size) {
		super();
		frame = new byte[size];
	}

	// check what kind of frame this is and return it
	static public LapFrame newFrame(byte[] iframe) {
		// unnumbered?
		if (((iframe[1] & 0x03) == 0x03) && ((iframe[0] & 0x01) == 0x01)) {
			if ((iframe[1] & 0xef) == XIDc) {
				return new LapXIDFrame(iframe);
			}
			if ((iframe[1] & 0xef) == (SNRMc & 0xff)) {
				return new LapSnrmFrame(iframe);
			}
		}
		if ((iframe[1] & 0x03) == 0x01) {
			return new LapSFrame(iframe);
		}

		if ((iframe[1] & 0x01) == 0x00) {
			return new LapIFrame(iframe);
		}

		// if nothing else then return generic frame
		return new LapFrame(iframe);
	}

	static public LapFrame newFrame(Frame iframe) {
		// create a new array with the data and call methode above
		int len = iframe.length;
		byte[] frame = new byte[len];
		byte[] org = iframe.data;
		for (int i = 0; i < len; i++) {
			frame[i] = org[i];
		}
		return newFrame(frame);
	}

	// method to check if this is a certain type of frame
	public boolean isUnnumbered() {
		return (frame[1] & 0x03) == 0x03;
	}

	public boolean isSupervisory() {
		return (frame[1] & 0x03) == 0x01;
	}

	public boolean isInformation() {
		return (frame[1] & 0x01) == 0x00;
	}

	// is this a control of response frame?
	public boolean isControl() {
		return (frame[0] & 0x01) == 0x01;
	}

	public boolean isReponse() {
		return (frame[0] & 0x01) == 0x00;
	}

	// Is Poll/Final set?
	public boolean isPoll() {
		return (frame[1] & 0x10) == 0x10;
	}

	public void setPoll() {
		frame[1] |= 0x10;
	}

	public boolean isFinal() {
		return (frame[1] & 0x10) == 0x10;
	}

	public void setFinal() {
		frame[1] |= 0x10;
	}

	// kinds of Unnumbered Controls
	public final static byte SNRMc = (byte) 0x83;
	public final static byte DISCc = (byte) 0x43;
	public final static byte XIDc = (byte) 0x2f;
	public final static byte UIDc = (byte) 0x03;

	public byte unnumberedControl() {
		// consider throwing exception when not command
		return (byte) (frame[1] & 0xef);
	}

	// kinds of Unnumbered responses
	public final static byte RNRMr = (byte) 0x83;
	public final static byte UAr = (byte) 0x63;
	public final static byte FRMRr = (byte) 0x87;
	public final static byte DMr = (byte) 0x0f;
	public final static byte RDr = (byte) 0x43;
	public final static byte UIr = (byte) 0x03;
	public final static byte XIDr = (byte) 0xbf;
	public final static byte TEST = (byte) 0xe3;

	public byte unnumberedResponse() {
		// consider throwing exception then not response frame
		return (byte) (frame[1] & 0xef);
	}

	public int getSlot() {
		return frame[12];
	}

	public int getMaxSlot() {
		int slots = (frame[11] & 0x03);
		switch (slots) {
			case 0:
						return 1;
			case 1:
						return 6;
			case 2:
						return 8;
		}
		return 16;
	}

	public boolean isAddressConflict() {
		return (frame[11] & 0x04) == 0x04;
	}

	public boolean isEndDiscovery() {
		return frame[12] == (byte) 0xff;
	}

	// Supervisory frame types
	public final static byte RR = (byte) 0x00;
	public final static byte RNR = (byte) 0x01;
	public final static byte REJ = (byte) 0x02;
	public final static byte SREJ = (byte) 0x03;

	public byte supervisoryControl() {
		return (byte) ((frame[1] & 0x0c) >>> 2);
	}

	public byte supervisoryResponse() {
		return (byte) ((frame[1] & 0x0c) >>> 2);
	}

	public byte getAddress() {
		return (byte) ((frame[0] >>> 1) & 0x7f);
	}

	public byte[] getFrame() {
		return frame;
	}
}

// vi: ts=3 sw=3 ai
