;-----------------------------------------------------------------------------
;	ADC_DAC-STEP.ASM
;	ADC and DAC test program for HSM/550
;
; Supports DS87C550 HighSpeed Micro on HSM/550 Development Board
;
; (C) Copyright 1999 Systronix, Inc All Rights Reserved
;               Systronix, Inc. Salt Lake City, Utah, USA
;               TEL: 801-534-1017  FAX:801-534-1019
;		www.systronix.com  email: support@systronix.com
;		COMPLETE SYSTEMS for RAPID PROTOTYPING
;
; DESCRIPTION ----------------------------------------------------------------
;
; HSM/550 DAC drives one or more ADC channels (you connect them on the board).
; All channels of the ADC are sampled.
; Output is on UART1 of the HSM/550.
; By pressing the space bar you step through a series of DAC outputs such as
; 1/8, 1/4, 3/8, etc of full scale. Moving a clip lead from the DAC output to
; each ADC input gives a quick indication of how accurate each ADC channel is.
; It uses no interrupts.
;
; With jumpers in the default positions, ADC input will use the external
; 2.048V reference and input range will be 0-4.096V. This matches the DAC
; output range of 0-4.096V. The 10-bit DAC output count is 0-4095 (FFF hex)
; and the 10-bit ADC input count is 0-1024 (03FF hex).
;
; Note: the DAC opamp on HSM/550 can drive a maximum of 4 loads to 4.096V,
; heavier loads will cause the voltage to drop off about about 3.750V.
;
; REVISION HISTORY -----------------------------------------------------------
;
; 99 Jun 18 bab	Removing C550 EQUs from ASM file, use controller def instead
; 99 Jun 08 bab Start, based on adc_dac1
;
;
; NOTES AND COMMENTS ---------------------------------------------------------
;
;
;-----------------------------------------------------------------------------
; Baud Rate
;
; For reliable communication the actual baud rate should be within +/- 3% of
; the ideal.
;
; Both the UARTs and Timers have different modes, so don't confuse the UART
; mode with the baud rate generator Timer mode. For our loader, we use UART1
; so that UART0 is left untouched. UART mode 1 is a 10-bit asynch mode
; compatible with RS232 systems. Mode 3 is used for multi-processor
; communication. UART modes 1 and 3 use identical baud rate generation.
;
; UART1 in mode 1 must use Timer1 as a baud rate generator. This is most
; easily done by placing Timer1 in its mode 2, 8-bit auto-reload mode.
; In this mode, when T1 overflows it automatically reloads with the value
; stored in TH1. TL1 does the actual counting. This mode requires no MCU
; overhead since it reloads automatically.
;
; UART Mode 1 and 3 baud rate for UART1 is
; 	baud = (2^SMOD / 32) * Timer1_overflow
; where Timer1 overflow = xtal / ( T1DIV * (256-TH1) )
; T1DIV controlled by T1M bit (CKCON.4), is 12(T1M=0) or 4 (T1M=1)
;
; With Timer1 in 8-bit autoreload mode, solving for TH1:
;	TH1 = 256 - ( (2^SMOD/32) * ( xtal / T1DIV * baud) )
;
; Setting SMOD effectively doubles the baud rate for a given TH1.
; The values below assume T1DIV = 4, and SMOD=1
;		--xtal-	-baud- 	-TH1-  	-error-
;		11.0592	57600	FDH	0
;			19200	F7H	0
;			 9600	EEH	0
;			
;		14.7456	115K	FEH	-0.17%
;			57600	FCH	0
;			38400	FAH	0
;			19200	F4H	0
;			 9600	E8H	0
;
;		22.1184	115K	FDH	-0.17%
;			57600	FAH	0
;			38400	F7H	0
;			19200	EEH	0
;			 9600	DCH	0
;
;	25.0000	38400	F6		-1.7%
;			19200	ECH	-1.7%
;			 9600	D7H	0.76
;
;	33.0000	115K	FCH		-11% forget it
;			57600	F7H	0.54
;			38400	F3H	-3.19% marginal
;			19200	E5H	0.54
;			 9600	CAH	0.54
;
; change the baud rate here in this equate

baud_rate       equ     0E5H 		; 33.0000, 19200
;baud_rate       equ     0F4H 		; 14.7456 19200
;baud_rate       equ     0EEH 		; 11.0592, 9600 or 22.1184 19200

;-----------------------------------------------------------------------------
;       ascii equates 
;-----------------------------------------------------------------------------
ctrlc   equ	003h               	; control c
bs      equ	008h               	; back space 
tab     equ	009h              	; horizontal tab, ^i 
lf      equ	00ah			; line feed
cr      equ	00dh			; carriage return
xon     equ	011h            	; rs232 protocol ^q
xoff    equ	013h			; rs232 protocol ^s
spa     equ	020h			; space
colon   equ	03ah			; ':'
amp	equ	040h			; '@
ques	equ	03fh			; '?'
slash	equ	02fh			; '/'
del	equ	07fh             	; delete character 

 
;------------------------------------------------------------------
;	Internal DATA used by program

; dac data is 12 bits, allowed values are 0-4095 (FFFH)
dac_datl	EQU	09H	; lsb (0-ff)	
dac_dath	EQU	0AH	; this is the msb (0-F)

; adc data format depends on the mode the ADC is in
adc_datl	EQU	0BH	; lsb (0-ff)	
adc_dath	EQU	0CH	; this is the msb (0-3)  

tmp		EQU	0DH	; used by output routines

;------------------------------------------------------------------
; Reset Vector
;------------------------------------------------------------------
	org	0h			; reset vector
	ajmp	start

					; can org other interrupt vectors here

start:
	mov	SP, #10H		; above ineternal data we're using
	orl	wdcon, #080H		; Turn on the SMOD bit for serial port 1
        mov     scon1, #050h            ; serial mode 1, 8 data, 1 start,stop
        mov     tmod, #021h             ; timers 0-16 bit; 1-8 bit reload
        mov	th1,  #baud_rate	; from baud rate calculation
        orl	ckcon, #010H		; set T1M so T1DIV is 4 instead of 12
        mov     tcon, #050h             ; timer on

        mov	scon, #050h
        orl	pcon, #80h		; set SMOD bit for 

sl_init:
        clr	a
	mov	ie,a			;Disable all interrupts.
	mov	eie, a			; c550 extended interrupts
        clr	ti_1		     	;Clear ri1, ti1 just to be safe.
        clr	ri_1
        clr	ri_0
        clr	ti_0

	mov	a, #1			; select uart1
  	mov	dptr,#banner		; uart1 message	
	lcall	putstr			; send it

	mov	a, #0			; select uart0
  	mov	dptr,#banner		; uart1 message	
	lcall	putstr			; send it 

;------------------------------------------------------------------
; Reference select here!
;
 	mov	PWMADR, #80h		; external ref
;	mov	PWMADR, #00h		; internal ref

	; scott used 0f5 as the value for channel 7 at 22 MHz
	; at 22.1184 MHz with CD1:0 = 10 (POR default) MCU clock is xtal/4
	; this is 180 nsec at 22.1184 MHz xtal
	; ADCON2(3:0) are APS(3:0) the prescaler
	; ADclk must be 1 usec - 6.25 usec
	; ADclk = 180 nsec (1 + APS(3:0))
	; a value of F for APS is a prescale of 16, so ADclk = 16x 180 nsec = 2.89 usec
	; a value of 5 is a little too small, 900 nsec.
	; a conversion takes 16 ADclks so for APS = F, that's 46 usec
	
	mov	ADCON2, #0ffh		; set outcf, mux, adc clock prescaler
	; mov	ADCON1, #0a2h		; set start, cont, ADON
	mov	ADCON1, #002h		; single mode, ADON
	
				 	
	mov	ADCON1, #022h		; set start, cont, ADON


;------------------------------------------------------------------
;	MAIN_LOOP
;
; step the DAC as we wish and call the adc scan and display routines.
; 

main_loop:
        mov	dac_datl, #0		; init dac_data 000
        mov	dac_dath, #0
	lcall	dac_write0
	lcall	dac_disp
	lcall	adc_scan


	; This is 1FF, 1/8 of full scale
	mov	dac_datl, #0FFH		; keep dac lsb at FF
	mov	dac_dath, #01H		; msb goes to 1
	lcall	dac_write0
	lcall	dac_disp
	lcall	adc_scan   

dac_step:
	; we started at 000, then went to 1FF, now go to 2FF, 5FF, etc	
	mov	a, #2			; amount to incr msb by
	add	a, dac_dath
	mov	dac_dath, a		; save the new value
	lcall	dac_write0
	lcall	dac_disp
	lcall	adc_scan 		; stay until user enters a char
          
        mov	a, dac_dath		; test dac msb
	cjne	a, #00fh, dac_step	; if not 0f in msb keep going

	; if here we finished scan of DAC over 0-4095 

	mov	dptr, #finished
	lcall	putstr			; send it	
	lcall	newline1	
	lcall	newline1

	ljmp	main_loop		; keep going

;------------------------------------------------------------------
;	ADC_SCAN
;
; scan all adc channels and display by calling adc_sample on
; each channel in sequence
; 
adc_scan:  
	mov	a, #1			; select uart1
  	mov	dptr,#adc_header	; uart1 message	
	lcall	putstr			; send it
	
adc_scan2:
	clr	ri_1
	jb	ri_1, adc_scan2
	
	mov	ADCON2, #0ffh		; channel 7
	lcall	adc_sample
	mov	a, #20h			; space char
	lcall	putch1
	
	mov	ADCON2, #0efh		; channel 6
	lcall	adc_sample
	mov	a, #20h			; space char
	lcall	putch1
	
	mov	ADCON2, #0dfh		; channel 5
	lcall	adc_sample
	mov	a, #20h			; space char
	lcall	putch1
	
	mov	ADCON2, #0cfh		; channel 4
	lcall	adc_sample
	mov	a, #20h			; space char
	lcall	putch1
	
	mov	ADCON2, #0bfh		; channel 3
	lcall	adc_sample
	mov	a, #20h			; space char
	lcall	putch1
	
	mov	ADCON2, #0afh		; channel 2
	lcall	adc_sample
	mov	a, #20h			; space char
	lcall	putch1
	
	mov	ADCON2, #09fh		; channel 1
	lcall	adc_sample
	mov	a, #20h			; space char
	lcall	putch1
	
	mov	ADCON2, #08fh		; channel 0
	lcall	adc_sample
	
	mov	a, #cr			; cr after last adc channel
	lcall	putch1 
	
	; now keep going unless we got an input char
	
	jnb	ri_1,  adc_scan2 
	
	lcall	newline1  
	lcall	newline1
	
	ret

;------------------------------------------------------------------
;	ADC_SAMPLE
;
; Sample one ADC channel, already addressed in the mux.
; Here we start a conversion, wait for it to complete,
; display and store the data in adc_datl and adc_dath,
; then return.
; 	

adc_sample:
	mov	ADCON1, #82h		; start conversion, clear EOC

adc_conv:	
	mov	A, ADCON1
	;clr	P1.1
	mov	C, acc.7
	mov	P1.0, C			; monitor BSY
	jnb	acc.6, adc_conv		; wait for conversion EOC

	;setb	P1.1			; adc eoc was 1
	;NOP
	;NOP

	mov	adc_datl, ADLSB
	mov	adc_dath, ADMSB
	mov	a, adc_dath		; output results to uart1
	lcall	putbyt1
	mov	a, adc_datl
	lcall	putbyt1

	ret

;------------------------------------------------------------------
;	DAC_DISP 
; Print the value of the DAC as 4 hex digits  
; And as X/8 of full scale

dac_disp: 
	mov	a, #1			; select uart1
	mov	dptr, #dac_val
	lcall	putstr			; send
		
	mov	a, dac_dath		; output results to uart1
	lcall	putbyt1
	mov	a, dac_datl
	lcall	putbyt1

	mov	a, #2Ch			; space
	lcall	putch1  	
	mov	a, #20h			; space
	lcall	putch1      
	
	mov	a, dac_dath		; dac msb
	inc	a			; add one
	clr	c
	rrc	a    			; divide by two 
	lcall	putbyt1 		; output dac value in x/8 of full scale
	mov	dptr, #dac_scale
	lcall	putstr			; send
	
	
	lcall	newline1
	
	ret




       
        

;------------------------------------------------------------------
;   data definitions
;------------------------------------------------------------------

banner:		db   	cr, cr
		db	'------------- HSM/550 TEST PROGRAM ADC_DAC_STEP -------------', cr, lf
		db	'         HSM/550 12-bit DAC drives 87C550 10-bit ADC', cr, lf
		db	'ADC input range and ext ref jumpers in default position.', cr, lf		
		db	'External ADC reference, DAC drives 1/8-of-full-scale steps.', cr, lf
		db	'Connect DAC output to one or more ADC inputs.', cr, lf 
		db	'note: DAC loads down when driving more than 4 ADC inputs.', cr, lf
		db	'DAC range is 0-FFF (12 bits), ADC is 0-3FF (10 bits)', cr, lf 
		db	'DAC is 1 mV per count, ADC is 4 mV per count.', cr, lf, lf
		db	'Press any key to move to the next step', cr, lf
		
		
;		db	'  (Refer to silkscreen on bottom of board for pinouts)', cr
;		db	'  Jumper all ADC range inputs 2-3 for 0-4.096V range', cr
;		db	'  DAC increments through all values @ 1 mV per count.', cr
;;		db	'  ADC and DAC use the same 2.048V precision reference.', cr
		db	cr, lf, 0

adc_header:	db	'ADC7 ADC6 ADC5 ADC4 ADC3 ADC2 ADC1 ADC0', cr, lf, 0

finished:	db   	'Finished with scan', 0 
dac_val:	db	'DAC value: ', 0
dac_scale:	db	'/8 of DAC full scale', 0



	INCL	"..\dac_550.inc"   
	INCL	"..\550_io.inc"
	


	END

