;-----------------------------------------------------------------------------
;	DS1284.INC
;
; (C) Copyright 1999 Systronix, Inc All Rights Reserved Internationally
;               Systronix, Inc. Salt Lake City, Utah, USA
;               TEL: 801-534-1017  FAX:801-534-1019
;		www.systronix.com  email: support@systronix.com
;		COMPLETE SYSTEMS for RAPID PROTOTYPING    

;
; DESCRIPTION ----------------------------------------------------------------
;
; Include file for the DS1284 Embedded Clock & Calendar (ECC) used in HSM/550
;
; REVISION HISTORY ----------------------------------------------------------- 
;
; 99 Jun 08 bab	Start, based on alc_1284.inc which was intended to be called
;		from BCI51. So in this file the BASIC code has been removed,
;		and this INC is usable just from assy code.
;
; Comments here apply to alc_1284.inc but may be of interest
;
; 98 Mar 21 BAB	Fixed an obscure bug in DS83_1224_WR which read in
;	the bcd hours, converted to hex, then didn't convert
;	back to bcd before writing new value to the ecc.
;	commented out unneeded code in _DS83_HR_RD24
;	added save of PSW to save carry flag in HEX_TO_BCD
;	Copied this fix from DS5001.SRL
;	Also fixed a bug in date_wr which was not present in DS5001
;	version. For some reason in this file I wasn't calling the
;	hex to bcd conversion, so date 21 decimal got stored as 15H
;	and then read back as 15 bcd. Oops.
;	Converted all the DS_83XXX labels to DS_84XXX
;
; 97 Dec 10 BAB	Adapting for DS1284 without data pins mis-ordered
; 97 Dec 08 BAB	Start, adapting for DS1283 on ALC?
;
;	USE: The start of a code segment to be retrieved must be a leading
;	underscore in the leftmost column.  If it is a "_%%", the label will 
;	be retrieved as a comment. This supports labeled groups of EQUates, 
;	etc which occur prior to an ORG and thus prior to the allowed use 
;	of labels.  Also this is how in-line code fragments are loaded, since
;	they appear repeatedly and cannot have duplicated labels.
;
;	Each code segment must end with a ";%%" sequence in the leftmost column.
;	This tells the compiler librarian where each library section ends.
;
;	In general, equates without a preceding underscore are numerical
;	constants.  Equates with a preceding underscore are addresses of
;	storage locations.
;
;<! ------------------------------------------------------------------------->
;<HR><H3> GENERAL NOTES </H3>
;
; Why not just read and write clock registers directly?
;
; Time is stored in BASIC variables such as DS1284_HOUR for hours. The actual
; value of hours is located in a clock chip register. But in the case of hours,
; other bits (6, and 5) have significance (12/24 hour and AMP/PM). Plus all the
; times are stored and counted in BCD, not HEX. So you can't simply read and
; write the hours clock register the way you might hope. To get around this
; complication, these routines mask out special bits, perform HEX to BCD
; conversion, and so forth.
;
;<! ------------------------------------------------------------------------->
;<HR><H3> USING THE DS1284, TIPS & ANOMALIES </H3>
;
; Jared discovered that the TE bit (0BH.7) must be set in order to modify the
; EOSC bit. This is apparently not in the Dallas data book.
;
; The DS1283 is VERY sensitive to stray capacitance. The DS1284 may not be as
; touchy. In any case you don't want any signal traces or ground or power
; planes very close to the crystal pins. Put voids in internal planes around
; the crystal pins. See Dallas app note 58 (www.dalsemi.com, data sheets and
; app notes ares) for all their tips on crystal considerations.
;
; The DS1283, 1284 and 1286 are, in my opinion, VERY different, much more so
; than the data sheets imply. In particular, the 1283 apparently has no write
; protection as VCC fails. The 1284 does. The 1283 is, again in my opinion,
; only useful when used with the soft micros DS5001 and DS5002.
;
; Add support for month and day strings
; Add support to set the AM/PM string
;
;-----------------------------------------------------------------------------
;
;-----------------------------------------------------------------------------
;
; Equates for DS1284 clock calendar chip
; These are the 1284 register locations which have the BCD values,
; not our local variables with binary/hexadecimal time values.
;

ECC_ADR		equ 0FEh	; DS1284 at FEXX (base address)

DS_84HSEC	EQU 00H		; hsec time
DS_84SEC	EQU 01H		; seconds time
DS_84MIN	EQU 02H		; minutes time
DS_84MIN_AL	EQU 03H		; minutes alarm
DS_84HOUR	EQU 04H
DS_84HOUR_AL	EQU 05H		; hours alarm
DS_84DAY	EQU 06H		; days 1-7
DS_84DAY_AL	EQU 07H		; days alarm
DS_84DATE	EQU 08H		; date 1-31
DS_84MONTH	EQU 09H
DS_84YEAR	EQU 0AH		; yrs 0-99
DS_84CMD	EQU 0BH		; command reg
DS_84INTV_HSEC	EQU 0CH		; wdog alarm hsec 0-99
DS_84INTV_SEC	EQU 0DH		; wdog alarm sec 0-99
;
; 0EH through 3FH are user NVRAM, but we use the first
; two locations for add'l alarm values
DS_84DATE_AL	EQU 0EH		; alarm date
DS_84MON_AL	EQU 0FH		; alarm month

; This leaves 10H - 3FH for user NVRAM

_R0	EQU	0       	; 
_R1	EQU	1		    
	    
;-----------------------------------------------------------------------------
;	Data Declarations
;
; This data resides in xdata space, it is not like the assy code equates above.
; Note that the assy code DB assigns constant data in CODE space.
; DS assigns changable data in DATA space.
;
 
PROG_CNTR	SET	$ 
 
	org	0  				; data space org


DS1284_HSEC:		DS	1
DS1284_SEC:		DS	1
DS1284_MIN:		DS	1
DS1284_MIN_AL:		DS	1
DS1284_HOUR:		DS	1
DS1284_HOUR_AL:		DS	1
DS1284_DAY:		DS	1
DS1284_DAY_AL:		DS	1
DS1284_DATE:		DS	1
DS1284_MONTH:		DS	1
DS1284_YEAR:		DS	1
DS1284_CMD:		DS	1
DS1284_INTV_HSEC:	DS	1
DS1284_INTV_SEC:	DS	1
DS1284_MASK_AL:		DS	1

DS1284_AMPM:		DS	1 		; iff 12 hr mode: 1=PM, 0=AM
DS1284_1224:		DS	1		; 12/24, 0 = 24 hr, non-0 = 12 hr
DS1284_RUN:		DS	1 		; 1 = run, 0 = halt clock
DS1284_SQW:		DS	1		; 1 = enable, 0 = halt sq wave output
DS1284_UPDATE:		DS	1  		; 1 = update, 0 = freeze updates

					; clock can still run if update=0
_DS1284_AMPM$:		DS	3  		; am/pm text string, 2 + null
_DS1284_DAY$:		DS	4  		; day text string, 3 + null
_DS1284_MONTH$:		DS	4  		; month text string, 3 + null

;-----------------------------------------------------------------------------
; This constant data table resides in code space

	ORG	PROG_CNTR

_DS_AMPM_TAB:	DB 'AMPM',0

_DS_DAY_TAB:	DB 'SunMonTueWedThuFriSat',0

_DS_MON_TAB:	DB 'JanFebMarAprMayJunJulAugSepOctNovDec',0



;<! ------------------------------------------------------------------------->
;<HR><H3> EMBEDDED CLOCK / CALENDAR SUPPORT ROUTINES </H3>
;
;
;

_%%DS1284:

;<!------------------------------------->
;<! CLOCK READ ROUTINES >
;<HR><H3><A NAME=CLOCK READS>CLOCK READS </A></H3>

_DS1284_HSEC_RD:
	mov	R1, #DS_84HSEC		; 1284 reg address
	mov	DPTR, #DS1284_HSEC 	; data storage location
	lcall	_DS1284_REG_RD2		; save, ret
	ret

_DS1284_REG_RD2:
	lcall	_DS1284_RB		; read an ECC byte to R0
	mov	A, R0			; put in acc
_DS1284_REG_RD3:
	lcall	_BCD_TO_HEX		; hex time in acc
_DS1284_REG_RD4:
	movx	@DPTR, A		; save acc value
	mov	R0, A			; save opreg0 lsb
	mov	R2, #0			; save opreg0 msb
	ret

_DS1284_SEC_RD:
	mov	R1, #DS_84SEC		; 1284 reg address
	mov	DPTR, #DS1284_SEC 	; data storage location
	lcall	_DS1284_REG_RD2		; save, ret
	ret

_DS1284_MIN_RD:
	mov	R1, #DS_84MIN		; 1284 reg address
	mov	DPTR, #DS1284_MIN 	; data storage location
	lcall	_DS1284_REG_RD2		; save, ret
	ret

_DS1284_DAY_RD:
	mov	R1, #DS_84DAY		; 1284 reg address
	mov	DPTR, #DS1284_DAY 	; data storage location
	lcall	_DS1284_REG_RD2		; get & save time
	ret

;---------------------------------------
; Not complete or used by compiler
;
_DS1284_DAY_RDS:			; read day string
					; read day integer

					; day in acc (1-7)
	mov	DPTR, #_DS_DAY_TAB	; day string table
	dec	A			; day now 0-6
	mov	B, #3			; length of each day str
	mul	AB			; calc offset

	; add to DPTR then call str_assn in array.srl
	ret

_DS1284_DATE_RD:
	mov	R1, #DS_84DATE		; 1284 reg address
	mov	DPTR, #DS1284_DATE 	; data storage location
	lcall	_DS1284_RB		; read an ECC byte to ACC and R0
	; in 1283, INP bit is MSB of date register, not part of the date
	; in 1284 & 1286 it's set always to zero, so make it zero in any case	
	clr	ACC.7			; inp never part of date
	lcall	_DS1284_REG_RD3		; conv bcd to hex, save, ret
	ret

_DS1284_MONTH_RD:
	mov	R1, #DS_84MONTH		; 1284 reg address
	mov	DPTR, #DS1284_MONTH 	; data storage location
	lcall	_DS1284_RB		; read an ECC byte to R0
	mov	A, R0
	anl	A, #01FH		; mask out osc & sqw
	lcall	_DS1284_REG_RD3		; save, ret
	ret

_DS1284_YEAR_RD:
	mov	R1, #DS_84YEAR		; 1284 reg address
	mov	DPTR, #DS1284_YEAR 	; data storage location
	lcall	_DS1284_REG_RD2		; save, ret
	ret

_DS1284_HOUR_RD:
	mov	R1, #DS_84HOUR		; 1284 reg address
	mov	DPTR, #DS1284_HOUR 	; data storage location
	lcall	_DS1284_RB		; read an ECC byte to R0
	mov	A, R0			; put in acc
	jnb	ACC.6, _DS1284_HR_RD24	; jump if 24 hr mode
	anl	_R0, #01FH		; mask out 12/24, AMPM bits
_DS1284_HR_RD24:
	anl	_R0, #03FH		; mask out 12/24 bit
	mov	A, R0			; put in acc
	lcall	_DS1284_REG_RD3		; conv to hex, save in data space
	ret

;-----------------------------------------------------------------------------
; 1224 = 12 or 24 for 12 or 24 hour mode

_DS1284_1224_RD:
	mov	R1, #DS_84HOUR		; 1284 reg address
	lcall	_DS1284_RB		; read an ECC byte to R0
	mov	A, R0			; put in acc
	mov	DPTR, #DS1284_1224 	; data storage location
	jb	ACC.6, _DS1284_1224_12	; jmp if 12 hr
	mov	A, #24
	sjmp	_DS1284_1224_A
_DS1284_1224_12:
	mov	A, #12
_DS1284_1224_A:
	lcall	_DS1284_REG_RD4		; save in data space, R0
	ret

;-----------------------------------------------------------------------------
; AMPM = 1 if PM and 12-hour mode, else 0

_DS1284_AMPM_RD:
	mov	R1, #DS_84HOUR		; 1284 reg address
	lcall	_DS1284_RB		; read an ECC byte to R0
	mov	A, R0			; put in acc
	mov	DPTR, #DS1284_AMPM 	; data storage location
	jb	ACC.6, _DS1284_AMPM_12	; jmp if 12 hr
	mov	A, #0
	sjmp	_DS1284_AMPM_A
_DS1284_AMPM_12:
	mov	C, ACC.5		; C=1 if PM
	mov	A, #0
	mov	ACC.0, C		; acc=1 if PM, 0 if AM
_DS1284_AMPM_A:
	lcall	_DS1284_REG_RD4		; save in data space, R0
	ret

;-----------------------------------------------------------------------------
;
_DS1284_CMD_RD:
	mov	R1, #DS_84CMD		; 1284 reg address
	mov	DPTR, #DS1284_CMD 	; data storage location
	lcall	_DS1284_RB		; read an ECC byte to R0
	mov	A, R0			; put in acc
	lcall	_DS1284_REG_RD4		; save in data space, R0
	ret

_DS1284_UPDATE_RD:
	lcall	_DS1284_CMD_RD		; get curr reg value
	mov	A, R0			; curr val in acc
	mov	DPTR, #DS1284_UPDATE 	; data storage location
	mov	C, ACC.7		; TE bit in C
	mov	A, #00H			; clear acc
	mov	ACC.0, C		; acc=1 if C set
	lcall	_DS1284_REG_RD4		; save in data space, R0
	ret	

;-----------------------------------------------------------------------------
; The 1284 EOSC and ESQW are asserted low to enable.
; BCI51 uses 1 to mean enabled, so we complement the raw bits.
; We combine both here to save code since both bits are adjacent.

_DS1284_RUN_RD:
	setb	F0			; 1=run read
	sjmp	_DS1284_RSQ_RD1
_DS1284_SQW_RD:
	clr	F0			; 0=sqw read

_DS1284_RSQ_RD1:
	mov	R1, #DS_84MONTH		; 1284 reg address
	lcall	_DS1284_RB		; read an ECC byte to R0
	mov	A, R0			; curr val in acc
	jnb	F0, _DS1284_SQW_RD1	; read sqw?
					; else read run

_DS1284_RUN_RD1:
	mov	C, ACC.7		; EOSC bit in C
	cpl	C
	mov	A, #0			; clear acc
	mov	ACC.0, C		; acc=1 if C set
	mov	DPTR, #DS1284_RUN 	; data storage location
	lcall	_DS1284_REG_RD4		; save in data space, R0
	sjmp	_DS1284_RSQ_RDX

_DS1284_SQW_RD1:
	mov	A, R0			; curr val in acc
	mov	C, ACC.6		; ESQW bit in C
	cpl	C
	mov	A, #0			; clear acc
	mov	ACC.0, C		; acc=1 if C set
	mov	DPTR, #DS1284_SQW 	; data storage location
	lcall	_DS1284_REG_RD4		; save in data space, R0

_DS1284_RSQ_RDX:
	ret



;-----------------------------------------------------------------------------
; Interval Timer

_DS1284_INTV_HSEC_RD:
	mov	R1, #DS_84INTV_HSEC	; 1284 reg address
	mov	DPTR, #DS1284_INTV_HSEC 	; data storage location
	lcall	_DS1284_REG_RD2		; save, ret
	ret

_DS1284_INTV_SEC_RD:
	mov	R1, #DS_84INTV_SEC	; 1284 reg address
	mov	DPTR, #DS1284_INTV_SEC 	; data storage location
	lcall	_DS1284_REG_RD2		; save, ret
	ret

;-----------------------------------------------------------------------------
; Time of Day Alarm

_DS1284_MIN_AL_RD:
	mov	R1, #DS_84MIN_AL	; 1284 reg address
	mov	DPTR, #DS1284_MIN_AL 	; data storage location
	lcall	_DS1284_RB		; read an ECC byte to R0
	anl	_R0, #07FH		; mask out alarm mask bit
	mov	A, R0			; put in acc
	lcall	_DS1284_REG_RD3		; conv to hex, save in data space
	ret

;------------------------------------------------
; Hour alarm has the same 12/24, AMPM as hours reg
; ??? Do we provide the same 1224, AMPM rd/wr???

_DS1284_HOUR_AL_RD:
	mov	R1, #DS_84HOUR_AL	; 1284 reg address
	mov	DPTR, #DS1284_HOUR_AL 	; data storage location
	lcall	_DS1284_RB		; read an ECC byte to R0
	mov	A, R0			; put in acc
	jnb	ACC.6, _DS1284_HR_AL_RD24	; jump if 24 hr mode
	anl	_R0, #01FH		; mask out al mask, 12/24, AMPM bits
_DS1284_HR_AL_RD24:
	; we jumped here if 12/24=0 (ie 24 hr mode) so bit 6 IS 0, no need
	; ...to mask it off here.
	; if we fell through, we masked out 12/24 and ampm so still no need
	; BAB 98 Mar 05 added here 98 Mar 21
;	anl	_R0, #03FH		; mask out 12/24 bit
	mov	A, R0			; put in acc
	lcall	_DS1284_REG_RD3		; conv to hex, save in data space
	ret

_DS1284_DAY_AL_RD:
	mov	R1, #DS_84DAY_AL	; 1284 reg address
	mov	DPTR, #DS1284_DAY_AL 	; data storage location
	lcall	_DS1284_RB		; read an ECC byte to R0
	anl	_R0, #07FH		; mask out alarm mask bit
	mov	A, R0			; put in acc
	lcall	_DS1284_REG_RD3		; conv to hex, save in data space
	ret

;-----------------------------------------------------------------------------
; Here we need to read the msb of min, hr and day alarm, and interpet thus:
;     min      hr     day
;     ---      --     ---			
;	1	1	1	alarm once per minute
;	0	1	1	alarm when min match
;	0	0	1	alarm when min & hr match
;	0	0	0	alarm when min & hr & day match
;
; We build up the value of 0-7 in the B reg, one bit at a time

_DS1284_MASK_AL_RD:
	mov	B, #0			; clear B reg
	mov	R1, #DS_84MIN_AL	; 1284 reg address
	lcall	_DS1284_RB		; read an ECC byte to R0
	mov	A, R0			; copy to acc
	mov	C, ACC.7
	mov	B.2, C

	mov	R1, #DS_84HOUR_AL	; 1284 reg address
	lcall	_DS1284_RB		; read an ECC byte to R0
	mov	A, R0			; copy to acc
	mov	C, ACC.7
	mov	B.1, C

	mov	R1, #DS_84DAY_AL	; 1284 reg address
	lcall	_DS1284_RB		; read an ECC byte to R0
	mov	A, R0			; copy to acc
	mov	C, ACC.7
	mov	B.0, C

	mov	DPTR, #DS1284_MASK_AL 	; data storage location
	mov	A, B			; copy read val to acc
	lcall	_DS1284_REG_RD4
	ret

;<!------------------------------------->
;<! CLOCK WRITE ROUTINES >
;<HR><H3><A NAME=CLOCK WRITES>CLOCK WRITES </A></H3>


_DS1284_MASK_AL_WR:
	mov	DPTR, #DS1284_MASK_AL 	; data storage location
	mov	A, R0			; copy to acc
	mov	B, A			; copy to B reg
	movx	@DPTR, A		; store new hex val in data space

	mov	C, ACC.2		; min alarm mask bit to C
	mov	R1, #DS_84MIN_AL	; 1284 reg address
	lcall	_DS1284_MASK_AL_WR2
	mov	C, ACC.1		; hr alarm mask bit to C
	mov	R1, #DS_84HOUR_AL	; 1284 reg address
	lcall	_DS1284_MASK_AL_WR2
	mov	C, ACC.0		; day alarm mask bit to C
	mov	R1, #DS_84DAY_AL	; 1284 reg address
	lcall	_DS1284_MASK_AL_WR2
	ret				; done with all bits

_DS1284_MASK_AL_WR2:
	push	ACC			; save acc (has new mask val)
	lcall	_DS1284_RB		; read an ECC byte to R0
	mov	A, R0			; copy to acc
	mov	ACC.7, C		; update alarm bit
	mov	R0, A			; save data in R0
	lcall	_DS1284_WB		; write ECC byte from R0
	pop	ACC
	ret				; done with this bit


_DS1284_HOUR_AL_WR:
	mov	R1, #DS_84HOUR_AL	; 1284 reg address
	mov	DPTR, #DS1284_HOUR_AL 	; data storage location
	ljmp	_DS1284_HOUR_WR2		; same as hours write

_DS1284_DAY_AL_WR:
	mov	R1, #DS_84DAY_AL	; 1284 reg address
	mov	DPTR, #DS1284_DAY_AL 	; data storage location
_DS1284_DAY_AL_WR2:
	mov	A, R0			; new hex value to write in acc
	movx	@DPTR, A		; store new hex val in data space
	lcall	_HEX_TO_BCD		; new bcd val in acc
	push	ACC			; save new bcd time
	lcall	_DS1284_RB		; read curr time BCD to R0
	anl	_R0, #80H		; mask all but alarm bit
	pop	ACC			; get back new BCD val
	orl	A, R0			; or in mask bit
	mov	R0, A			; save data in R0
	lcall	_DS1284_WB		; write ECC byte from R0
	ret

_DS1284_MIN_AL_WR:
	mov	R1, #DS_84MIN_AL	; 1284 reg address
	mov	DPTR, #DS1284_MIN_AL 	; data storage location
	sjmp	_DS1284_DAY_AL_WR2	; same as day al from here


;-----------------------------------------------------------------------------
; Writing to the DS1284 Timekeeper Registers and our local data variables
;-----------------------------------------------------------------------------

_DS1284_HSEC_WR:
	mov	R1, #DS_84HSEC		; 1284 reg address
	mov	DPTR, #DS1284_HSEC 	; data storage location
	lcall	_DS1284_REG_WR2		; calc and store it
	ret

_DS1284_REG_WR2:
	mov	A, R0			; hex value to write in acc
	movx	@DPTR, A		; store hex val in data space
	lcall	_HEX_TO_BCD		; bcd hrs in acc
	mov	R0, A			; save data in R0
	lcall	_DS1284_WB		; write ECC byte from R0
	ret

_DS1284_REG_WRH:
	mov	A, R0			; hex value to write in acc
	movx	@DPTR, A		; store hex val in data space
	lcall	_DS1284_WB		; write ECC byte from R0
	ret


_DS1284_SEC_WR:
	mov	R1, #DS_84SEC		; 1284 reg address
	mov	DPTR, #DS1284_SEC 	; data storage location
	lcall	_DS1284_REG_WR2		; calc and store it
	ret

_DS1284_MIN_WR:
	mov	R1, #DS_84MIN		; 1284 reg address
	mov	DPTR, #DS1284_MIN 	; data storage location
	lcall	_DS1284_REG_WR2		; calc and store it
	ret

;-----------------------------------------------------------------------------
; In 24 hour mode, the hour reg holds only 0-23 hours
; In 12 hour mode we leave 12/24 and AMPM bits as is, logically OR in the new 
; time.
; We share this routine with the hour alarm write also, so we preserve
; the alarm mask bit 7.
; Call here with new value to write (Hex) already in R0

_DS1284_HOUR_WR:
	mov	R1, #DS_84HOUR		; 1284 reg address
	mov	DPTR, #DS1284_HOUR 	; data storage location
_DS1284_HOUR_WR2:
	mov	A, R0			; new hex value to write in acc
	movx	@DPTR, A		; store new hex val in data space
	lcall	_HEX_TO_BCD		; new bcd val in acc
	push	ACC			; save new bcd time

	lcall	_DS1284_RB		; read curr time BCD to R0
	mov	A, R0
	jnb	ACC.6, _DS1284_HRWR_24	; if acc.6=0, 24 hr mode
	anl	A, #0E0H		; mask out current time...
					; ...keep alarm, 1224, ampm bits
	pop	B			; new BCD time in B
	orl	A, B			; or in new time
	sjmp	_DS1284_HRWR_A		; store new time
_DS1284_HRWR_24:
	mov	C, ACC.7		; save mask bit if set
	pop	ACC			; get back new BCD time
	mov	ACC.7, C		; restore mask bit
_DS1284_HRWR_A:
	mov	R0, A			; save data in R0
	lcall	_DS1284_WB		; write ECC byte from R0
	ret


;-----------------------------------------------------------------------------
; AMPM = 1 if PM and 12-hour mode, else we don't change anything

_DS1284_AMPM_WR:
	push	_R0			; save new AMPM value
	mov	R1, #DS_84HOUR		; 1284 reg address
	mov	DPTR, #DS1284_AMPM 	; data storage location
	lcall	_DS1284_RB		; read curr time BCD to R0
	mov	A, R0			; curr time in ACC
	jnb	ACC.6, _DS1284_APWR_24	; if acc.6=0, 24 hr mode
	pop	B
	mov	C, B.0
	mov	ACC.5, C		; new AMPM bit in curr time
	movx	@DPTR, A		; save data value
	mov	R0, A			; new data also in R0
	lcall	_DS1284_WB		; write ECC byte from R0
	ret

_DS1284_APWR_24:
	pop	B			; waste saved value
	mov	A, #0			; 24 hr mode so...
	movx	@DPTR, A		; store 0 in AMPM data space
	ret


_DS1284_DAY_WR:
	mov	R1, #DS_84DAY		; 1284 reg address
	mov	DPTR, #DS1284_DAY 	; data storage location
	lcall	_DS1284_REG_WR2		; calc and store it
	ret

_DS1284_DATE_WR:
	mov	R1, #DS_84DATE		; 1284 reg address
	mov	DPTR, #DS1284_DATE 	; data storage location
	lcall	_DS1284_REG_WR2		; calc and store it
	ret

_DS1284_YEAR_WR:
	mov	R1, #DS_84YEAR		; 1284 reg address
	mov	DPTR, #DS1284_YEAR 	; data storage location
	lcall	_DS1284_REG_WR2		; calc and store it
	ret

;-----------------------------------------------------------------------------
; We logically OR in the new months with the current EOSC and ESQW bits,
; leaving them unchanged.

_DS1284_MONTH_WR:
	mov	R1, #DS_84MONTH		; 1284 reg address
	mov	DPTR, #DS1284_MONTH 	; data storage location
	mov	A, R0			; new hex value to write in acc
	movx	@DPTR, A		; store new hex val in data space
	lcall	_HEX_TO_BCD		; new bcd value in acc
	push	ACC			; save new bcd time

	lcall	_DS1284_RB		; read curr time BCD to R0
	anl	_R0, #0C0H		; mask off curr time
	pop	ACC	
	orl	_R0, A			; or in new time to R0
	lcall	_DS1284_WB
	ret

;-----------------------------------------------------------------------------
; If new 1224 value is 24, we clr the 1224 flag, if <24 we set it
; so that the DS1284 1224 flag=0 if 24 hr, 1 if 12 hr mode

_DS1284_1224_WR:
	push	_R0			; save new value
	lcall	_DS1284_HOUR_RD		; get hrs reg value in R0
	pop	ACC		   	; new val in acc, curr hrs in R0
	cjne	A, #24, _DS1284_1224_WR1	; C set if acc < 24
_DS1284_1224_WR1:
	mov	A, R0			; curr hrs in acc
	lcall	_HEX_TO_BCD		; convert hours in acc back to bcd	
	mov	ACC.6, C		; update 1224 bit in acc
	mov	R0, A			; new value also in R0
	lcall	_DS1284_WB		; save new value
	lcall	_DS1284_1224_AL_WR	; set alarm 12/24 bit to match
	ret

;-----------------------------------------------------------------------------
; This routine sets the 12/24 bit in the hours alarm register
; Call with the C bit set for 12 hour mode or clear for 24 hour

_DS1284_1224_AL_WR:
	mov	R1, #DS_84HOUR_AL	; 1284 reg address
	lcall	_DS1284_RB		; read an ECC byte to R0
_DS1284_1224_AL_WR1:
	mov	A, R0			; curr hrs in acc
	mov	ACC.6, C		; update 1224 bit in acc
					; other bits unchanged
	mov	R0, A			; new value also in R0
	lcall	_DS1284_WB		; save it
	ret

_DS1284_UPDATE_WR:
	mov	R1, #DS_84CMD		; 1284 reg address
	mov	A, R0			; new val in acc
	push	ACC			; save new value
	lcall	_DS1284_RB		; read curr val into R0
	pop	ACC			; new val in acc
	mov	C, ACC.0		; C = new update value
	mov	A, R0			; curr value in acc
	mov	ACC.7, C		; set new TE bit val
	mov	DPTR, #DS1284_UPDATE 	; data storage location
	movx	@DPTR, A		; store new val in data space
	mov	R0, A			; also in R0
	lcall	_DS1284_WB		; write new cmd reg value
	ret

_DS1284_RUN_WR:
	mov	R1, #DS_84MONTH		; 1284 reg address
	mov	DPTR, #DS1284_RUN 	; data storage location
	mov	A, R0			; new val in acc
	movx	@DPTR, A		; store new val in data space
	push	ACC			; save new value
	lcall	_DS1284_RB		; read curr val into R0
	pop	ACC			; new val in acc
	mov	C, ACC.0		; C = new update value
	cpl	C			; 1= run but 0 = EOSC(L)
	mov	A, R0			; curr value in acc
	mov	ACC.7, C		; set new EOSC bit val
	mov	R0, A			; also in R0
	lcall	_DS1284_WB		; write new cmd reg value
	ret

_DS1284_SQW_WR:
	mov	R1, #DS_84MONTH		; 1284 reg address
	mov	DPTR, #DS1284_SQW 	; data storage location
	push	_R0			; save new value
	mov	A, R0
	movx	@DPTR, A		; store new val in data space
	lcall	_DS1284_RB		; read curr val into R0
	pop	ACC			; new val back in acc
	mov	C, ACC.0		; C = new update value
	cpl	C			; 1= run but 0 = EOSC(L)
	mov	A, R0			; curr value in acc
	mov	ACC.6, C		; set new EOSC bit val
	mov	R0, A			; also in R0
	lcall	_DS1284_WB		; write new cmd reg value
	ret

_DS1284_CMD_WR:
	mov	R1, #DS_84CMD		; 1284 reg address
	mov	DPTR, #DS1284_CMD 	; data storage location
	lcall	_DS1284_REG_WRH		; store it as hex
	ret


;-----------------------------------------------------------------------------
; Interval Timer

_DS1284_INTV_HSEC_WR:
	mov	R1, #DS_84INTV_HSEC	; 1284 reg address
	mov	DPTR, #DS1284_INTV_HSEC 	; data storage location
	lcall	_DS1284_REG_WR2		; calc and store it
	ret	

_DS1284_INTV_SEC_WR:
	mov	R1, #DS_84INTV_SEC	; 1284 reg address
	mov	DPTR, #DS1284_INTV_SEC 	; data storage location
	lcall	_DS1284_REG_WR2		; calc and store it
	ret


;-----------------------------------------------------------------------------
; Convert BCD byte in acc to hex value in acc.
;
; B is saved, used, restored. R0 is used
;
; MSN = high nibble of BCD, LSN = low nibble
; HEX = (MSN * 10) + LSN
;
; No error checking is done.  We assume acc holds a valid BCD value.

_BCD_TO_HEX:
	push	B
	mov 	R0, A			; copy acc
	anl	A, #0F0H 		; clear low nibble
	swap	A			; high nibble now in low place
	mov 	B, #10			; mul old high nibble by 10
	mul	AB			; old high nib * 10 in acc
	mov	B,A			; partial result to B
	xch	A, R0			; old byte back in acc
	anl	A, #00FH 		; mask off old high nibble
	add	A, B			; add old low nibble to acc
	pop	B
	ret



;-----------------------------------------------------------------------------
; Convert hex byte in acc to BCD value in acc.
;
; Value in acc is 0-99 decimal, otherwise results are invalid.
;
; BCD = HEX/10 + HEX MOD 10
;
; No error checking is done.  We assume acc holds a value which can be 
; correctly converted to a BCD value.
;

_HEX_TO_BCD:
	push	PSW			; save carry flag
	push	B
	mov	B, #10
	div	AB			; acc = acc /10, B = acc rem 10
	swap	A
	add	A,B
	pop	B
	pop	PSW	
	ret


;-----------------------------------------------------------------------------
; RB: Read a byte from 1284 to R0
; location to read is in R1
; DPTR is preserved to same value it was at entry
;
; We don't clear the TE bit here since we only access 1 byte location, if
; we did access multiple bytes then TE should be disabled to maintain time
; synchronization.  For example if you tried to read 11:59 and the hours ticked
; over to 12 just after you read the 59, you'd read the time incorrectly as
; 12:59 when it's really 12:00.

_DS1284_RB:
	push	DPL			; save DPTR
	push	DPH
	push 	IE
	anl	IE, #82H		; leave EA and ET0 for RTC
	mov	DPH, #0FEH		; FEXX for ALC
	mov	DPL, R1			; reg addr to read
	movx	A, @DPTR		; time value in acc
	mov	R0, A			; save time value in R0
	pop	IE			; restore interrupts
	pop	DPH
	pop	DPL
	ret


;-----------------------------------------------------------------------------
; Write a byte from R0 to 1284
;
; 1284 reg addr is in R1, data to write is in R0
; DPTR value at entry is not used here, but it is overwritten here.
;
; We don't clear the TE bit here since we only access 1 byte location, see
; comments in DS1284_RB above.

_DS1284_WB:
	push 	IE
	anl	IE, #82H		; leave EA and ET0 for RTC
	mov	DPH, #0FEH		; FEXX for ALC
	mov	DPL, R1			; reg addr to read
	mov	A, R0			; value to set in acc
	movx	@DPTR, A		; store time value
	pop	IE			; restore interrupts
	ret

;%%





read_time:
	; read clock and cal time, this is a temporary hack
	lcall	_DS1284_HSEC_RD
	lcall	_DS1284_SEC_RD
	lcall	_DS1284_MIN_RD
	lcall 	_DS1284_HOUR_RD
	lcall	_DS1284_DAY_RD
	lcall	_DS1284_DATE_RD
	lcall	_DS1284_MONTH_RD
	lcall	_DS1284_YEAR_RD
	lcall	_DS1284_1224_RD
	lcall	_DS1284_AMPM_RD
	lcall	_DS1284_CMD_RD
	lcall	_DS1284_UPDATE_RD
	lcall	_DS1284_RUN_RD

	ret

