;-----------------------------------------------------------------------------
;	550_IO.INC
;
; (C) Copyright 1999 Systronix, Inc All Rights Reserved
;               Systronix, Inc. Salt Lake City, Utah, USA
;               TEL: 801-534-1017  FAX:801-534-1019
;		www.systronix.com  email: support@systronix.com
;		COMPLETE SYSTEMS for RAPID PROTOTYPING    

;
; DESCRIPTION ----------------------------------------------------------------
;
; Common serial I/O routines for the HSM550. It is an include file with no
; crystal or time dependencies. The serial ports must be set up elsewhere.
;
; REVISION HISTORY ----------------------------------------------------------- 
;
; 99 Jun 08 bab	taking out CR to CR-LF combination in putch0 and putch1 so that
;		we can keep output on one line.  
;		Added newline1 and newline0 routines
; 99 Jun 07 bab	start
;
;
;
; NOTES AND COMMENTS ---------------------------------------------------------
;
; PUTSTR     	take the string pointed to by dptr and prints it out uart0 or 1
; PUTCH1	take the char in the accumulator, send it out uart0
; PUTCH0  	take the char in the accumulator, send it out uart0
; PUTBYT1	Take the binary value in acc and put out two hex digits to uart1  
; NEWLINE[0|1]	outputs a CR-LF pair on UART0 or UART1
;
;-----------------------------------------------------------------------------
  

;-----------------------------------------------------------------------------
; Internal Data declarations
; Make sure other parts of your application don't conflict with these!
  
tmp     	equ	0DH		; temp used: putbyt


;-----------------------------------------------------------------------------
;	PUTSTR
; 
; take the string pointed to by dptr and prints it out. 
; the string must be terminated by a null.
; dptr is left pointing at the null terminator.
; acc holds 0 if output goes to uart0, nonzero otherwise and
; output will be sent to uart1
; 
; inputs: dptr, acc (0 or nonzero)
; outputs: none 
; variables modified: a, dptr 
; error exits: none 
; subroutines accessed directly: putch
;-----------------------------------------------------------------------------
putstr:
	jz	putstr0			; send to uart0 if acc is 0...
					; ... else send to uart1
putstr1:
        clr  	a                	; make address offset zero 
        movc 	a,@a+dptr       	; get next char to print 
        jz   	puts_x          	; when zero we are at end of string
        lcall 	putch1     		; write it out 
        inc 	dptr              	; point to next char 
        ljmp 	putstr1           	; do it again
        sjmp	puts_x

putstr0:
        clr  	a                	; make address offset zero 
        movc 	a,@a+dptr        	; get next char to print 
        jz   	puts_x           	; when zero we are at end of string
        lcall 	putch0      		; write it out 
        inc   	dptr             	; point to next char 
        ljmp  	putstr0          	; do it again
puts_x:
        ret	

;-----------------------------------------------------------------------------
;	PUTCH1
; 
; take the char in the accumulator, send it out uart1
; wait til the char is sent
; Uses no interrupts

putch1:
        mov   	sbuf1,a           	; copy a to sbuf for uart1
        jnb     ti_1,$  			; wait till char is gone from uart1 
                                                                      
clr_ti1:
        clr	ti_1			; C550 rev A2 serial bug means we must...
        jb	ti_1, clr_ti1  		; ...keep clearing until it's really cleared

; C550 Rev A2 needs delay between clearing TI and writing to SBUF
        NOP				
        NOP
        NOP
	NOP
        NOP
        NOP  
                                 
putch1_x:
        ret

;-----------------------------------------------------------------------------
;	PUTCH0
; 
; take the char in the accumulator, send it out uart0
; wait til the char is sent
;

putch0:
        mov   	sbuf,a           	; copy a to sbuf for uart0
        jnb     ti_0,$  			; wait till char is gone from uart0
                
clr_ti0:
        clr	ti_0			; C550 rev A2 serial bug means we must...
        jb	ti_0, clr_ti0		; ...keep clearing until it's really cleared
        
; C550 Rev A2 needs delay between clearing TI and writing to SBUF        
        NOP
        NOP
        NOP
	NOP
        NOP
        NOP           
                               
putch0_x:
        ret


;-----------------------------------------------------------------------------
;	PUTBYT1
; 
; Take the binary value in a and put out two hex digits to uart1
; 
;   inputs: a 
;   outputs: none 
;   variables modified: a, tmp 
;   error exits: none 
;   subroutines accessed directly: putch1
;-----------------------------------------------------------------------------

putbyt1:
        mov    tmp,a                    ; save the byte to be printed (push) 
        swap   a                        ; shift high nibble over 
        anl    a,#0fh                   ; now we really have just a nibble 
        add    a,#90h                   ; 90h-99h,9ah-9fh
        da     a                        ; 90h-99h,00h-05h
        addc   a,#40h                   ; d0h-d9h,41h-46h
        da     a                        ; 30h-39h,41h-46h which is 0-9,a-f ascii
        lcall  putch1                    ; and now its on its way.
  
        mov    a,tmp                    ; get the low nibble now 
        anl    a,#0fh                   ; now we really have just a nibble 
        add    a,#90h                   ; 90h-99h,9ah-9fh
        da     a                        ; 90h-99h,00h-05h
        addc   a,#40h                   ; d0h-d9h,41h-46h
        da     a                        ; 30h-39h,41h-46h which is 0-9,a-f ascii
        lcall  putch1                   ; and now its on its way too.
  
        ret 
        
;-----------------------------------------------------------------------------
;	newline1
; Put a CR-LF pair out to UART0 or 1
;        
        
newline1:
	mov	a, #cr			; newline
	lcall	putch1
	mov	a, #lf			
	lcall	putch1			
	ret

newline0:
	mov	a, #cr			; newline
	lcall	putch0
	mov	a, #lf			
	lcall	putch0			
	ret


;-----------------------------------------------------------------------------
; Convert BCD byte in acc to hex value in acc.
;
; B is saved, used, restored. R0 is used
;
; MSN = high nibble of BCD, LSN = low nibble
; HEX = (MSN * 10) + LSN
;
; No error checking is done.  We assume acc holds a valid BCD value.

BCD_TO_HEX:
	push	B
	mov 	R0, A			; copy acc
	anl	A, #0F0H 		; clear low nibble
	swap	A			; high nibble now in low place
	mov 	B, #10			; mul old high nibble by 10
	mul	AB			; old high nib * 10 in acc
	mov	B,A			; partial result to B
	xch	A, R0			; old byte back in acc
	anl	A, #00FH 		; mask off old high nibble
	add	A, B			; add old low nibble to acc
	pop	B
	ret



;-----------------------------------------------------------------------------
; Convert hex byte in acc to BCD value in acc.
;
; Value in acc is 0-99 decimal, otherwise results are invalid.
;
; BCD = HEX/10 + HEX MOD 10
;
; No error checking is done.  We assume acc holds a value which can be 
; correctly converted to a BCD value.
;

HEX_TO_BCD:
	push	PSW			; save carry flag
	push	B
	mov	B, #10
	div	AB			; acc = acc /10, B = acc rem 10
	swap	A
	add	A,B
	pop	B
	pop	PSW	
	ret


