;;-----------------------------------------------------------------------------
;;
;; LCD INCLUDE FILE for DPB2 boards rev D and rev E
;;
;;
;;	(C) Copyright 1995 Systronix, Inc
;;	All rights reserved internationally.
;;	Systronix, Inc. Salt Lake City, Utah, USA
;;	TEL: 801-534-1017  FAX:801-534-1019  BBS:801-487-2778
;;
;; Owners of Systronix products are hereby entitled to use, modify, and copy
;; this sample program.  You may incorporate all or part of this program in 
;; your own commercial products provided they are not simply copies of 
;; Systronix products and do not compete directly with Systronix products.  
;; This program is provided by Systronix free of charge and may not be resold.
;;
;; This program is intended for those of you who have the pleasure of using 
;; the new DPB2 REVE board from Systronix/Intellix.  This card is a very
;; easy to use development system for the Dallas DS5000/1/2 DS2250/1/2 and -T
;; suffix soft microcontrollers.  If you'd like more information, please call
;; or FAX us at the numbers above.
;;
;; REVISION HISTORY LCD_E.INC -------------------------------------------------
;;
;; 95 Aug 13 	BAB	Added lcd_clear_linex for 20x4, can be modified for 
;;			shorter or longer lines by changing the BASIC 
;;			variable LCD_CHARS.
;;			Also fixed a longstanding bug in userdpb2.srl which
;;			caused a conflict between serial I/O and print@ to LCD.
;;			See notes in USERDPB2.SRL
;;
;; 95 Jul 18	BAB	Replaced all interrupt disables of EX1 with EA
;;			This is necessary when an ext int0 handler uses
;;			P0 bits.
;;	95 Jan 30	BAB	Fixed some shift direction bugs and return
;;				pop mismatch reported by DH
;;	94 Nov 28	BAB	Cleaned up tabs/spaces, removed commented-out code
;;	03/02/94	BAB	Fixed blank display bug and LCD/ONEX1 collisions
;;				with simultaneous PORT0 use.
;;				Disabled ONEX1 interrupt in LCD access
;;	01/25/94	BAB	Cleanup, comments
;;	11/XX/94	KDH	Start
;;
;;
;; NOTES AND COMMENTS ---------------------------------------------------------
;;
;; Keep all printing to the LCD or serial port outside interrupt handlers.  
;; If you have print routines interrupting other print
;; routines you can expect all sorts of strange-looking output. Alternatively 
;; you can have all printing inside one interrupt handler.  The point is to 
;; keep printing in one place and don't have print routines interrupting each 
;; other!
;;
;; The keypad on DPB2 shares Port0 with the LCD data bus.  Therefore we turn
;; off EX1 (used by ONEX1, the keypad handler) during an LCD write.  Otherwise,
;; a keypress could interrupt an LCD data transfer and place keypad data on the
;; LCD data bus.  Guaranteed to cause strange-looking LCD display!
;;
;;-----------------------------------------------------------------------------
;
; LCD Access Timing:
;
; LCD initialization occurs automaticlly provided:
;   1) Power supply rise from 0.2 to 4.5V is 0.1 msec min, 10 msec max
; 	2) Power supply off (less than 0.2V) time is at least 1 msec
;	During power-on init, the busy flag is held high, typically for 10 msec
;
; If there is any chance the power supply doesn't meet these conditions, you
; should reset by firmware instructions.	
;
; LCD intialization by firmware is different from normal instruction writes, 
; in that you cannot monitor the busy flag and delay between writes is up to 
; 4.1 msec.
; 
;
; Instruction writes to the LCD take 40 usec max except for clear display and
; return home which take up to 1.64 msec.  You can monitor the busy flag for
; all these.
;
;-----------------------------------------------------------------------------
;	95 Jan	A customer has reported that Rohm brand LCDs do not use the
; same initialization timing as all the other LCDs we have used (Denistron,
; Optrex, Kyocera, Samsung, Hitachi, Samtron, etc).
;
;
;-----------------------------------------------------------------------------
; HD44780 and equivalent character type LCD controller description
;
; Hitachi HD44780 type controllers have up to 80 character locations, hex
; addresses 00H thru 4FH (4FH = 80 decimal).  When right-shifting even
; an 8-character display, location 4FH shifts into character 0!
;
; DD RAM = Display Data RAM, character codes 20H thru FFH
; CG RAM = Character Generator RAM, character codes 00H thru 0FH
;
;
;-----------------------------------------------------------------------------
; LCD Display Sizes and Addresses
;
; One line displays:
;	 8x1 has DD RAM addresses 00H thru 07H, left to right
;	16x1 has DD RAM addresses 00H thru 0FH, left to right
;
; Two line displays, less than 40x2:
;	Top line has DD RAM addresses 00H thru 27H, left to right
;	Bottom line has DD RAM addresses 40H thru 67H, left to right
;
;	A right shift wraps each line on itself - address 27H moves into the 
;	top line, character 0.  Address 67H moves into the bottom line first
;	character.
;	
;	
;
;
;-----------------------------------------------------------------------------
;
; LCD CONTROL
; 
; Entry points:
;   lcd_init        -- initialize the lcd
;   lcd_clr_display -- clear the entire lcd display
;   lcd_home        -- set the lcd display to the original position,
;                        (undo display shift) and set cursor postion to (0,0)
;   lcd_cursor_inc  -- set cursor to write from left to right
;   lcd_cursor_dec  -- set cursor to write from right to left
;   lcd_shift_on    -- set display to shift as if cursor stays stationary
;   lcd_shift_off   -- set display to not shift
;   lcd_display_on  -- turn the display on
;   lcd_display_off -- turn the display off
;   lcd_cursor_on   -- turn the cursor on
;   lcd_cursor_off  -- turn the cursor off
;   lcd_blink_on    -- turn the cursor blink on
;   lcd_blink_off   -- turn the cursor blink off
;   lcd_goto_line1  -- set the cursor position to (0,1)
;   lcd_goto_line2  -- set the cursor position to (0,2)
;   lcd_goto_line3  -- set the cursor position to (0,3)
;   lcd_goto_line4  -- set the cursor position to (0,4)
;   lcd_goto_xy     -- set the cursor position to (lcd_xpos,lcd_ypos)
;   lcd_set_dd_pos  -- set the cursor position to absolute value to lcd_dat
;   lcd_get_dd_pos  -- get the cursor's absolute value postion into lcd_dat
;   lcd_write_data  -- write the data in lcd_dat to the current position
;   lcd_read_data   -- get the data into lcd_dat from the current position
;
;    lcd_clear_line1 (also 2,3,4)
;
; P1.4 is LCD RS (register select), Data (H), Instruction (L)
; P1.5 is LCD R/W (read/write) Read (H), Write (L)
; P1.6 is LCD enable, pulse it H to strobe the cycle into the LCD
;
; P0 is the data bus
; Busy flag is data bit 7, P0.7
;
;
	goto lcd_inc_end                ; protect against include in program body

	unsigned char lcd_dat
	unsigned char lcd_busy
	signed int lcd_xpos
	signed int lcd_ypos

	unsigned char lcd_disp_stat	; display on/off, cursor on/off, blink on/off
	unsigned char lcd_em_stat	; cursor move right/left, display shift on/off

					; number of chars per LCD row
	unsigned char LCD_CHARS		; 20 for 20x4 LCD, 16 for 16x2, etc

;-----------------------------------------------------------------------------
;
; Applies to any resolution display

lcd_init:
;    print "lcd_init"
	lcd_em_stat = 0
	lcd_disp_stat = 0

	clock1
	msec = 0 : do : while (msec < 20)			; delay 20 msecs

	LCD_DAT = 030H						; function set 8 bits
	gosub lcd_ins_wrnb
	msec = 0 : do : while (msec < 5)			; delay 5 msecs

	LCD_DAT = 030H						; function set 8 bits
	gosub lcd_ins_wrnb
	msec = 0 : do : while (msec < 5)			; delay 5 msecs

	LCD_DAT = 030H						; function set 8 bits
	gosub lcd_ins_wrnb
	msec = 0 : do : while (msec < 5)			; delay 5 msecs

	; set interface to 8 bits, 1 line display
	; 030H = 8 bits, one line, 5x7 font
	; 038H = 8 bits, 2 or 4 lines, 5x7 font
	LCD_DAT = 038H
	gosub lcd_ins_wrnb

	gosub lcd_clr_display
	gosub lcd_display_on

	if LCD_CHARS = 0 then LCD_CHARS = 20			; default

	return


; Clear display, cursor home (takes about 2 msec)
; Be sure to call with long delay version of instr write
lcd_clr_display:
	LCD_DAT = 01H
	goto lcd_ins_wrnb

; Resets display to original position and cursor position to 0,  (takes about 2 msec)
; Be sure to call with long delay version of instr write
lcd_home:
	LCD_DAT = 02H
	goto lcd_ins_wrnb

;-----------------------------------------------------------------------------
; Cursor move and display shift are performed during data read and write
; These apply to any resolution display

REM Set Cursor shift to right after a data read or write
lcd_cursor_inc:
	lcd_em_stat = lcd_em_stat or 0000_0010B			; set shift direction bit
	goto lcd_em_las

REM Set Cursor shift to left
lcd_cursor_dec:
	lcd_em_stat = lcd_em_stat and 1111_1101B		; clear shift direction bit
	goto lcd_em_las

REM Set display shift on after a data read or write
lcd_shift_on:
	lcd_em_stat = lcd_em_stat or 0000_0001B			; set display shift bit
	goto lcd_em_las

REM Set display shift off
lcd_shift_off:
	lcd_em_stat = lcd_em_stat and not 0000_0001B		; clear display shift bit
	goto lcd_em_las

; Entry mode is a write cycle which sets cursor move and display shift
;
REM load current entry mode status, set entry mode control bit and strobe
lcd_em_las:
	LCD_DAT = lcd_em_stat or 0000_0100B			; set entry mode control bit
	goto lcd_ins_write


;-----------------------------------------------------------------------------
; Display on/off, cursor on/off and blink on/off
; These apply to any resolution display

REM Turn on display
lcd_display_on: 
	lcd_disp_stat = lcd_disp_stat or 0000_0100B		; set display bit
	goto lcd_disp_las

REM Turn off display
lcd_display_off:
	lcd_disp_stat = lcd_disp_stat and not 0000_0100B	; clear display bit
	goto lcd_disp_las

REM Turn on cursor
lcd_cursor_on:
	lcd_disp_stat = lcd_disp_stat or 0000_0010B		; set cursor bit
	goto lcd_disp_las

REM Turn off cursor
lcd_cursor_off:
	lcd_disp_stat = lcd_disp_stat and not 0000_0010B	; clear cursor bit
	goto lcd_disp_las

REM Turn on blink of cursor location
lcd_blink_on:
	lcd_disp_stat = lcd_disp_stat or 0000_0001B		; set blink bit
	goto lcd_disp_las

REM Turn off blink
lcd_blink_off:
	lcd_disp_stat = lcd_disp_stat and not 0000_0001B	; clear blink bit
	goto lcd_disp_las

; Now do a display status write
REM load current display status, set display control bit and strobe
lcd_disp_las:
	LCD_DAT = lcd_disp_stat or 0000_1000B			; set display control bit
	goto lcd_ins_write


;-------------------------------------
;	LCD CLEAR ROUTINES
;-------------------------------------

lcd_clear_line1:
	gosub lcd_goto_line1
	goto clear_line

lcd_clear_line2:
	gosub lcd_goto_line2
	goto clear_line

lcd_clear_line3:
	gosub lcd_goto_line3
	goto clear_line

lcd_clear_line4:
	gosub lcd_goto_line4
	goto clear_line

clear_line:
	;	12345678901234567890
#ASM
	mov	DPTR, #__LCD_CHARS			; 20 for 20x4 LCD, etc
	movx	A, @DPTR				; # of chars per line
	mov	R0, A
	mov	A, #20H					; space char in acc
_CLEAR_MORE:
	lcall	_PRINT_AT				; output space to clear LCD char
	djnz	R0, _CLEAR_MORE				; repeat LCD_CHARS times
	ret
#ASM_END




;-----------------------------------------------------------------------------
;	GO TO SPECIFIC LCD DD RAM ADDRESSES
;
; Used with care these apply to any resolution.
;

REM set to 1st row addr=000H, 1st line (0,1)
lcd_goto_line1:
	lcd_dat = 000H
	goto lcd_set_dd_pos

REM set to 2nd line (0,2)
lcd_goto_line2:
	lcd_dat = 040H
	goto lcd_set_dd_pos

REM set to 2nd row addr=040H, 3rd line (0,3)
lcd_goto_line3:
	lcd_dat = 014H
	goto lcd_set_dd_pos

REM set to 4th line (0,4)
lcd_goto_line4:
	lcd_dat = 054H
	goto lcd_set_dd_pos

REM set the cursor position to (lcd_xpos,lcd_ypos)
lcd_goto_xy:
;	print "lcd_goto_xy"
	if (lcd_ypos = 4) then
        lcd_dat = 054H
    else
        if (lcd_ypos = 3) then
        	lcd_dat = 014H
        else
            if (lcd_ypos = 2) then
                lcd_dat = 040H
            else
                lcd_dat = 000H
            endif
        endif
    endif
    if (lcd_xpos >= 1) and (lcd_xpos <= 20) then
        lcd_dat = lcd_dat + lcd_xpos - 1
    endif
    goto lcd_set_dd_pos


REM set the DD RAM pointer to the contents of lcd_dat
lcd_set_dd_pos:
    	LCD_DAT = lcd_dat or 080H                             ; set DD RAM control bit
    	goto lcd_ins_write

REM get the DD RAM pointer to the contents in lcd_dat
lcd_get_dd_pos:
#ASM
	push	IE
	clr	EA
	CLR     P1.4	    				; set RS low for instr
	SETB    P1.5	    				; set RD/WR high for read
	MOV     P0, #0FFH				; make PO input
	lcall	_LCD_RD_STROBE
	mov	A, P0					; get data from LCD
	mov	DPTR, #__LCD_DAT
	movx	@DPTR, A				; store the data
	pop 	IE					; restore interrupts
#ASM_END

    return

;-----------------------------------------------------------------------------
;	lcd_write_data	
; write the data in LCD_DAT to the LCD, disable EX1 during the write

lcd_write_data:
#ASM
	mov	DPTR, #__LCD_DAT			; point to data we want
	movx	A, @DPTR				; data in acc
	push	IE					; save IE
	clr 	EA					; turn off EX1
	mov	P0, A					; move LCD_DAT to port0
#ASM_END
lcd_data_write:         				; BASIC call here with PORT0 = data
#ASM
_lcd_data_write:					; assy call here with PORT0 = data
	SETB    P1.4	    				; set RS high for data
	CLR     P1.5	    				; set RD/WR low for write
	ljmp	_LCD_STROBE
#ASM_END

;-----------------------------------------------------------------------------
;	LCD_READ_DATA
; Reads data from the character generator (CG) or display data (DD) RAM.
; The Address of the read must have been set by 1) a previous address set
; instruction or 2) a cursor shift instruction (applies to DD RAM only).
; A prior CG or DD write is not sufficient!

REM read data from the LCD into BASIC variable LCD_DAT
lcd_read_data:
#ASM
	push	IE					; save IE
	clr 	EA					; turn off EX1
	mov	P0, #0FFH				; make PO input
	SETB    P1.4	    				; set RS high for data
	SETB    P1.5	    				; set RD/WR high for read
	
	lcall	_LCD_RD_STROBE

	mov	A, P0					; get data from LCD
	mov	DPTR, #__LCD_DAT
	movx	@DPTR, A				; store the data
	pop 	IE					; restore interrupts
#ASM_END
	return


REM instruction write
lcd_ins_write:          				; call here with LCD_DAT = data
#ASM
	mov	DPTR, #__LCD_DAT			; point to data we want
	movx	A, @DPTR				; data in acc
	push	IE					; save IE
	clr 	EA					; turn off interrupts
	mov	P0, A					; move LCD_DAT to port0
	CLR     P1.4	    				; set RS low for instr
	CLR     P1.5	    				; set RD/WR low for write
	ljmp	_LCD_STROBE
#ASM_END


lcd_ins_wrnb:          					; call here with LCD_DAT = data
							; "no busy flag version" of an 
							; instruction write
#ASM
	mov	DPTR, #__LCD_DAT			; point to data we want
	movx	A, @DPTR				; data in acc
	push	IE					; save IE
	clr 	EA					; turn off EX1
	mov	P0, A					; move LCD_DAT to port0
	CLR     P1.4	    				; set RS low for instr
	CLR     P1.5	    				; set RD/WR low for write
	ljmp	_LCD_STROBE_NB
#ASM_END




REM data read strobe instruction returns with data in LCD_DAT
lcd_data_read:
#ASM
_lcd_data_read:						; assy call here
	SETB    P1.4	    				; set RS high for data
	SETB    P1.5	    				; set RD/WR high for read
#ASM_END
    goto lcd_strobe

;-----------------------------------------------------------------------------
;	LCD_STROBE
; 
; Strobe the data on PORT0 to the LCD interface chip.
; This is the exit point for most LCD routines.

lcd_strobe:
#ASM
_LCD_STROBE:
	CLR     P1.6	    				; clear enable
	SETB    P1.6	    				; strobe it
	CLR     P1.6	    				; clear enable
	SETB    P1.5	    				; set RD(H)/WR(L) high

;	LJMP	_LCD_BSY_CHK				; check busy flag
	LJMP	_LCD_BSY_TIME				; time instead of busy flag

_LCD_STROBE_NB:						; longer 1.64 msec wait
	CLR     P1.6	    				; clear enable
	SETB    P1.6	    				; strobe it
	CLR     P1.6	    				; clear enable
	SETB    P1.5	    				; set RD(H)/WR(L) high
	; wait about 2 msec at 12 MHz
	; Use this when you can't use the busy flag (LCD init)
	mov	R0, #4

_LCD_STROBEL2:
	mov	B, #255
	DJNZ	B, $
	DJNZ	R0, _LCD_STROBEL2

	pop	IE					; restore interrupts
	ret
	
#ASM_END

;-----------------------------------------------------------------------------
;	_LCD_BSY_CHK
; Note that if busy flag stays high execution will wait here forever.
; This could happen if the LCD is unpowered or not connected.

#ASM
_LCD_BSY_CHK:
	mov	P0, #0FFH				; make P0 input
	CLR	P1.4					; set R/S low
	SETB 	P1.5					; set RD/WR high
	SETB	P1.6					; strobe it w/ high enable
;	CLR	P2.0					; debug: watch with scope probe to time busy
	JB	P0.7, $					; wait while busy
	CLR	P1.6					; clear enable
;	SETB	P2.0					; drive pin back to normal high
	pop 	IE					; restore interrupts
	ret


;-----------------------------------------------------------------------------
;	_LCD_BSY_TIME
;
; Use this instead of LCD_BSY_CHK if LCD will not always be connected
;


_LCD_BSY_TIME:
	mov	B,#40
	DJNZ	B,$
	pop 	IE					; restore interrupts
	ret

#ASM_END

;-----------------------------------------------------------------------------
;	LCD_RD_STROBE
; This is a strobe and delay used in reading data from the LCD
; We don't pop IE here since we still have to store the read data after ret.
; This differs from the write strobe routine which is complete once the strobe
; cycle ends.  In a read, the strobe cycle gets the data on P0.

#ASM
_LCD_RD_STROBE:
	CLR     P1.6	    				; clear enable
	SETB    P1.6	    				; strobe it
	CLR     P1.6	    				; clear enable
	; wait about 80 usec
	; this could also poll the busy flag, better for extended temp range
	mov	B,#40
	DJNZ	B,$
	ret
#ASM_END







lcd_inc_end:                    			; protect against include in program body
